package ec2

import (
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/ec2"
	"log"
)

// Takes an array of tags, and returns the DescribeInstancesInput needed to search for EC2 Instances
func GetSearchInputFromTags(tags map[string]string, maxResults int64) *ec2.DescribeInstancesInput {
	return &ec2.DescribeInstancesInput{
		Filters:    buildSearchFilters(tags),
		MaxResults: &maxResults,
	}
}

// Tags an array of tags, and converts that to EC2 Filters
func buildSearchFilters(tags map[string]string) []*ec2.Filter {
	var filters []*ec2.Filter

	// Create a filter for each tag
	for name, value := range tags {
		filters = append(filters, buildFilterFromTag(name, value))
	}

	// Only include running instances
	// TODO: Make this optional
	filters = append(filters, &ec2.Filter{Name: aws.String("instance-state-name"), Values: []*string{aws.String("running")}})
	return filters
}

// Takes a single tag and converts it to an EC2 Filter
func buildFilterFromTag(tagName string, tagValue string) *ec2.Filter {
	filterName := "tag:" + tagName
	filterValue := tagValue

	return &ec2.Filter{
		Name: aws.String(filterName),
		Values: []*string{
			aws.String(filterValue),
		},
	}
}

// Retrieves the Instances from a DescribeInstancesOutput result
func getInstancesFromResult(result *ec2.DescribeInstancesOutput) []*ec2.Instance {
	var instances []*ec2.Instance
	if result == nil { return instances }

	// Go through all the reservations and grab the instances from within it
	for _, reservation := range result.Reservations {
		for _, instance := range reservation.Instances {
			instances = append(instances, instance)
		}
	}

	return instances
}

// Returns an EC2 Instance Array as pointers to the instance ids
func getInstanceIDsAsPointers(instances []*ec2.Instance) []*string {
	var awsInstanceIDList[]*string
	for _, instance := range instances {
		awsInstanceIDList = append(awsInstanceIDList, aws.String(*instance.InstanceId))
	}
	return awsInstanceIDList
}

// Returns an EC2 Instance array as strings to the instance ids
func GetInstanceIDsAsStrings(instances []*ec2.Instance) []string {
	var awsInstanceIDList []string
	for _, instance := range instances {
		awsInstanceIDList = append(awsInstanceIDList, *instance.InstanceId)
	}
	return awsInstanceIDList
}

// Makes sure we're actually restarting the right instances! This is a safety check. Looks at multiple things
func InstanceSearchIsValid(instances []*ec2.Instance) bool {
	validFlag := true

	for _, instance := range instances {
		if serviceTagMatches(instance) {
		} else {
			log.Printf("The following instance did not pass validation: %s", instance)
			validFlag = false
			return false
		}
	}

	return validFlag
}

// Ensures the instances is a part of the Grid Service, based on the tag Service:grid, for validation safety
func serviceTagMatches(instance *ec2.Instance) bool {
	serviceTagName := "Service"
	serviceTagValue := "grid" // Purposely hardcoded as a precaution
	found := false // Default to false, incase it gets through all tags and didn't find a Service tag

	// Look Through All Tags
	for _, tag := range instance.Tags {
		if *tag.Key == serviceTagName { // If it has a tag called Service
			found = *tag.Value == serviceTagValue
			return found
		}
	}
	return found
}

// Takes a list of EC2 Instances and returns a link to the EC2 Dashboard with a list of all the instances
func GetEC2ResultsURL(instances []*ec2.Instance, region string) string {
	baseURL := "https://us-west-2.console.aws.amazon.com/ec2/v2/home?region=" + region
	var instanceStr, instanceID string

	for _, instance := range instances {
		instanceID = *instance.InstanceId

		// If there are no instances yet in the list, assign it to the first. Else, append it
		if len(instanceStr) == 0 {
			instanceStr = instanceID
		} else { // append
			instanceStr = instanceStr + "," + instanceID
		}
	}

	return fmt.Sprintf("%s#Instances:instanceId=%s;sort=tag:Name", baseURL, instanceStr)
}

// Prints an AWS Error
func printAWSError(err error) {
	if err == nil {
		return
	}

	if awsErr, ok := err.(awserr.Error); ok {
		log.Println(awsErr.Error())
	}
}
