package ec2

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/stretchr/testify/assert"
	"log"
	"testing"
)


var testInstanceID = "i-test12345"
var testServiceTagValue = "grid"

func TestInstanceSearchIsValid(t *testing.T) {
	// Mock the instances
	instances := []*ec2.Instance {createTestInstance(testServiceTagValue),}

	// Make the call
	actual := InstanceSearchIsValid(instances)

	// Good Condition
	assert.Equal(t, true, actual)

	// Bad Condition - Break Something
	instances = []*ec2.Instance {createTestInstance("not-grid"),}
	actual = InstanceSearchIsValid(instances)
	assert.Equal(t, false, actual)
}

func TestBuildFilterFromTag(t *testing.T) {
	tagName := "Service"
	tagValue := "Test"

	filter := buildFilterFromTag(tagName, tagValue)

	assert.Equal(t, string(*filter.Name), *aws.String("tag:" + tagName))
	assert.Equal(t, len(filter.Values), 1, "Expected only one value per tag")
	assert.Equal(t, string(*filter.Values[0]), tagValue)
}

func TestGetSearchInputFromTags(t *testing.T) {
	tags := createTestTags()
	maxResults := int64(10)

	filters := buildSearchFilters(tags)
	actual := GetSearchInputFromTags(tags, maxResults)

	expected := &ec2.DescribeInstancesInput {
		Filters: filters,
		MaxResults: &maxResults,
	}

	assert.Equal(t, actual.MaxResults, expected.MaxResults)

	// Make sure all of the expected filters are within the actual filter array
	for _, value := range actual.Filters {
		assert.Contains(t, expected.Filters, value)
	}
}

func TestBuildSearchFilters(t *testing.T) {
	tags := createTestTags()
	expected := createTestFilters(tags)
	actual := buildSearchFilters(tags)

	// Ensure each of the expected filters are contained within the actual filter array
	for _, value := range actual {
		assert.Contains(t, expected, value)
	}
}

func TestAppendInstancesFromResult(t *testing.T) {
	var actual []*ec2.Instance
	instance := createTestInstance(testServiceTagValue)

	expected := []*ec2.Instance {
		instance,
	}

	output := createTestDescribeInstanceOutput(createTestReservation(instance))

	actual = append(actual, getInstancesFromResult(output)...)
	assert.Equal(t, expected, actual)
}

func TestGetInstanceIDsAsStrings(t *testing.T) {
	instances := []*ec2.Instance {
		createTestInstance(testServiceTagValue),
	}

	expected := []string {
		testInstanceID,
	}

	actual := GetInstanceIDsAsStrings(instances)

	assert.Equal(t, expected, actual)
}

func createTestTags() map[string]string {
	var tags = make(map[string]string)
	tags["Service"] = "Test"
	tags["Environment"] = "Dev"
	return tags
}

func createTestFilters(tags map[string]string) []*ec2.Filter {
	var filters []*ec2.Filter
	var filter *ec2.Filter

	if len(tags) <= 0 {
		log.Fatal("Problem creating test tags. Length was 0")
	}

	for name, value := range tags {
		filter = &ec2.Filter{
			Name: aws.String("tag:" + name),
			Values: []*string{
				aws.String(value),
			},
		}

		filters = append(filters, filter)
	}

	instanceRunningFilter := createFilterForInstanceState("running")
	filters = append(filters, instanceRunningFilter)

	return filters
}

func createFilterForInstanceState(state string) *ec2.Filter {
	return &ec2.Filter{Name: aws.String("instance-state-name"), Values: []*string{aws.String(state)}}
}

func createTestInstance(serviceTagValue string) *ec2.Instance {
	// Attach a "Service" Tag to it
	tagName := "Service"
	tag := ec2.Tag{
		Key: &tagName,
		Value: &serviceTagValue,
	}
	tags := []*ec2.Tag {&tag,}

	// Create the Instance
	instance := ec2.Instance{
		InstanceId: &testInstanceID,
		Tags: tags,
	}

	return &instance
}

func createTestReservation(instance *ec2.Instance) *ec2.Reservation {
	instances := []*ec2.Instance {instance,}
	res := ec2.Reservation{Instances: instances,}
	return &res
}

func createTestDescribeInstanceOutput(reservation *ec2.Reservation) *ec2.DescribeInstancesOutput {
	reservations := []*ec2.Reservation {reservation,}
	output := ec2.DescribeInstancesOutput{Reservations: reservations,}
	return &output
}
