// Contains code related to Rebooting Nodes
package grid_reboot

import (
	"code.justin.tv/qe/grid_reboot/pkg/config"
	"code.justin.tv/qe/grid_reboot/pkg/ec2"
	"errors"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	awsec2 "github.com/aws/aws-sdk-go/service/ec2"
	"log"
)

/*
Searches EC2 for all of the hubs in the config, and return their instance ids
Note: Not using a shared method, as the logic for getting the complete list of hubs is a little different (multiple name tags)
 */
func GetHubs(gridConfig config.GridRebootConfig, ec2Service *ec2.Handler) ([]*awsec2.Instance, error) {
	// Search individually for each hub
	instances, err := ec2Service.GetInstances(gridConfig.EC2.Hub.Tags, 100)
	if err != nil {
		return instances, err
	}

	// Print out the hubs to stdout
	log.Printf("Retrieved these Hubs: %s\n\nAWS URL:\n%s\n",
		ec2.GetInstanceIDsAsStrings(instances),
		ec2.GetEC2ResultsURL(instances, gridConfig.EC2.Region))

	return instances, nil
}

// Fetches nodes by their cluster
func GetNodesByCluster(clusterID string, ec2Service *ec2.Handler, gridConfig *config.GridRebootConfig) ([]*awsec2.Instance, error) {
	// Create the tags for searching
	tags := gridConfig.EC2.Nodes.Tags
	if tags == nil { // If no tags are in the config, make a map to hold them
		tags = make(map[string]string, 0)
	}
	tags["GridClusterID"] = clusterID // append the cluster id to the tags

	return ec2Service.GetInstances(tags, 500)
}

// Reboots nodes by their cluster
func RebootNodesByCluster(clusterID string, ec2Service *ec2.Handler, gridConfig *config.GridRebootConfig) error {
	instances, err := GetNodesByCluster(clusterID, ec2Service, gridConfig)
	if err != nil { return err }

	return rebootInstances(instances, ec2Service)
}

/*
Reboots a list of instances
 */
func rebootInstances(instances []*awsec2.Instance, ec2Service *ec2.Handler) error {
	if len(instances) <= 0 {
		log.Println("No instances found. Returning")
		return nil
	}

	// Make sure every instance about to be rebooted is a Grid instance
	if !ec2.InstanceSearchIsValid(instances) {
		return errors.New("invalid instance found, skipping the reboot")
	}

	logMsg := "Rebooting instances: "
	for _, instance := range instances {
		logMsg = fmt.Sprintf("%s %s", logMsg, aws.StringValue(instance.InstanceId))
	}
	log.Println(logMsg)
	
	_, err := ec2Service.RebootInstances(instances)
	return err
}

