# Contains configuration for ECS Tasks

# ECS Task that will run Grid Reboot
resource "aws_ecs_task_definition" "GridReboot" {
  family                = "Grid_Reboot"
  container_definitions = "${data.template_file.GridRebootContainer.rendered}"
  task_role_arn = "${aws_iam_role.GridRebootTask.arn}"
  network_mode = "awsvpc"
  execution_role_arn = "${aws_iam_role.GridRebootExecution.arn}"
  cpu = "256" // 0.25 vcpu
  memory = "512" // 0.5 GB
  requires_compatibilities = ["FARGATE"]
}

# Container Definition that will run as an ECS Task
# https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html
data "template_file" "GridRebootContainer" {
  template = "${file("${path.module}/container.json")}"
  vars = {
    docker_registry = "${var.ecs_repository_url}"
    image_name = "grid_reboot"
    image_version = "latest"
    cloudwatch_log_group  = "${aws_cloudwatch_log_group.GridReboot.id}"
    grid_router_host      = "${var.grid_router_host}"
    grid_hub_environment  = "${var.grid_hub_environment}"
    grid_node_environment = "${var.grid_node_environment}"
  }
}

# Create a place to store the logs
resource "aws_cloudwatch_log_group" "GridReboot" {
  name = "/aws/ecs/grid-reboot"
  retention_in_days = 30
  tags = {
    Owner = "dylan"
  }
}

# Triggers for the Task

# Run on a Cron Basis
resource "aws_cloudwatch_event_rule" "GridRebootTrigger" {
  name        = "grid-reboot-trigger"
  description = "Reboots Grid on a cron basis"
  schedule_expression = "cron(${var.reboot_cron_expression})"
  is_enabled = true
}

# The Target for the event (run the task)
resource "aws_cloudwatch_event_target" "GridRebootECS" {
  rule      = "${aws_cloudwatch_event_rule.GridRebootTrigger.name}"
  arn       = "${aws_ecs_cluster.GridReboot.arn}"
  role_arn = "${aws_iam_role.GridRebootECSEventsRole.arn}"
  ecs_target {
    task_definition_arn = "${aws_ecs_task_definition.GridReboot.arn}"
    task_count = 1
    launch_type = "FARGATE"
    network_configuration {
      subnets = ["${var.subnets}"]
      security_groups = ["${aws_security_group.grid_reboot.id}", "${data.aws_security_group.grid_router_vpce.id}"]
      assign_public_ip = false
    }
  }
}

# Security group for the container
resource "aws_security_group" "grid_reboot" {
  name        = "grid_reboot_sg"
  description = "Grid Reboot Container SG"
  vpc_id      = "${var.vpc_id}"

  // Allow container to communicate with anything, but don't let inbound connections in
  egress {
    from_port       = 0
    to_port         = 0
    protocol        = "-1"
    cidr_blocks     = ["0.0.0.0/0"]
  }

  tags {
    Name = "grid_reboot_sg"
  }
}
