# Contains configuration for Task Policies

# Responsible for Launching Fargate Tasks
resource "aws_iam_role" "GridRebootExecution" {
  name = "GridRebootTaskExecutionRole"
  assume_role_policy = <<EOF
{
  "Version": "2008-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# The built in ECS Task Execution Policy
data "aws_iam_policy" "AmazonECSTaskExecutionRolePolicy" {
  arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_iam_role_policy_attachment" "GridRebootAttach" {
  role = "${aws_iam_role.GridRebootExecution.name}"
  policy_arn = "${data.aws_iam_policy.AmazonECSTaskExecutionRolePolicy.arn}"
}

# Allows for triggering of the ECS Events
# Without this, Cloudwatch Events can't trigger pagerduty events on the SNS Topic
resource "aws_iam_role" "GridRebootECSEventsRole" {
  name = "GridRebootECSEventsRole"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "events.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

data "aws_iam_policy" "AmazonEC2ContainerServiceEventsRole" {
  arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceEventsRole"
}

resource "aws_iam_role_policy_attachment" "GridRebootECSEventsAttach" {
  role = "${aws_iam_role.GridRebootECSEventsRole.name}"
  policy_arn = "${data.aws_iam_policy.AmazonEC2ContainerServiceEventsRole.arn}"
}

# The Role that the Task Container actually uses
resource "aws_iam_role" "GridRebootTask" {
  name = "GridRebootTaskRole"
  assume_role_policy = <<EOF
{
  "Version": "2008-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# Allows for Describing EC2 Instances
resource "aws_iam_policy" "ec2ReadPolicy" {
  name        = "GridRebootEC2ReadPolicy"
  path        = "/"
  description = "Allows to describe instances for Grid Reboot"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1548800004357",
      "Action": [
        "ec2:DescribeInstanceStatus",
        "ec2:DescribeInstances"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "ec2ReadAttach" {
  role = "${aws_iam_role.GridRebootTask.name}"
  policy_arn = "${aws_iam_policy.ec2ReadPolicy.arn}"
}

# Allows for Rebooting EC2 Instances only of Service: grid
resource "aws_iam_policy" "ec2RebootPolicy" {
  name = "GridRebootEC2RebootPolicy"
  path = "/"
  description = "Allows to reboot instances for Grid Reboot only by tagged instances Service: grid"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "VisualEditor0",
      "Effect": "Allow",
      "Action": "ec2:RebootInstances",
      "Resource": "*",
      "Condition": {
        "StringLike": {
            "ec2:ResourceTag/Service": "grid"
        }
      }
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "ec2RebootAttach" {
  role = "${aws_iam_role.GridRebootTask.name}"
  policy_arn = "${aws_iam_policy.ec2RebootPolicy.arn}"
}

# Allows for Describing and Enabling/Disabling of Cloudwatch Event Roles, which triggers Lambda
resource "aws_iam_policy" "events_policy" {
  name        = "GridRebootCloudwatchEventsPolicy"
  path        = "/"
  description = "Allows describing, enabling and disabling rules"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1548800076482",
      "Action": [
        "events:DescribeRule",
        "events:DisableRule",
        "events:EnableRule"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "GridRebootEventsAttach" {
  role = "${aws_iam_role.GridRebootTask.name}"
  policy_arn = "${aws_iam_policy.events_policy.arn}"
}

# Allow Grid Reboot to report things to Cloudwatch Metrics like its Execution Result
resource "aws_iam_policy" "metric_policy" {
  name = "GridRebootPutCloudwatchMetrics"
  path = "/"
  description = "Allows Grid Reboot to put metrics to Cloudwatch for its results"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1548895743996",
      "Action": [
        "cloudwatch:PutMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "GridRebootMetricPutAttach" {
  role = "${aws_iam_role.GridRebootTask.name}"
  policy_arn = "${aws_iam_policy.metric_policy.arn}"
}

# Sandstorm Configuration
# policy that allows your role to assume your sandstorm role
data "aws_iam_policy_document" "sandstorm" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]
    resources = [
      "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/qe-prod-grid"
    ]
  }
}

resource "aws_iam_role_policy" "sandstorm_policy" {
  name = "grid-reboot-sandstorm-policy"
  role = "${aws_iam_role.GridRebootTask.id}"
  policy = "${data.aws_iam_policy_document.sandstorm.json}"
}

resource "aws_iam_policy" "beanstalk_policy" {
  name = "GridRebootBeanstalkPolicy"
  path = "/"
  description = "Allows Grid Reboot to restart Beanstalk and describe data"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1611814550637",
      "Action": [
        "elasticloadbalancing:RegisterInstancesWithLoadBalancer",
        "autoscaling:DescribeAutoScalingGroups",
        "cloudformation:DescribeStackResource",
        "elasticbeanstalk:DescribeEnvironmentHealth",
        "elasticbeanstalk:DescribeEnvironments",
        "elasticbeanstalk:RestartAppServer",
        "elasticbeanstalk:RetrieveEnvironmentInfo"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "beanstalkAttach" {
  role = "${aws_iam_role.GridRebootTask.name}"
  policy_arn = "${aws_iam_policy.beanstalk_policy.arn}"
}
