// This package stores Global App Configuration that multiple packages will need
package config

import (
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor"
	"fmt"
	"github.com/jonboulle/clockwork"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"os"
	"strings"
	"time"
)

const (
	logLevelEnvName = "LOG_LEVEL"
	defaultForwardRequestTimeout = time.Minute
)

type Config struct {
	Logger         *logrus.Logger
	Clock          clockwork.Clock
	Instrumentor   *instrumentor.Instrumentor
	ForwardRequestTimeout time.Duration
}

// Creates a new configuration object
func New() (*Config, error) {
	config := &Config{
		Logger: logrus.New(),
		Clock:  clockwork.NewRealClock(),
		ForwardRequestTimeout: defaultForwardRequestTimeout,
	}
	config.Logger.SetLevel(getLogLevel())
	return config, nil
}

// Returns a config that should only be used in tests
func NewMock() *Config {
	logger, _ := test.NewNullLogger()
	return &Config{
		Logger: logger,
		Clock: clockwork.NewFakeClock(),
		ForwardRequestTimeout: defaultForwardRequestTimeout,
	}
}

// Returns a log level to use based on the environment variable logLevelEnvName
// No input will default to Info
// Unrecognized input will return Info
func getLogLevel() logrus.Level {
	var logLevel logrus.Level

	// toLower - case insensitive input
	switch requestedLevel := strings.ToLower(os.Getenv(logLevelEnvName)); requestedLevel {
	case "debug":
		logLevel = logrus.DebugLevel
	case "info":
		logLevel = logrus.InfoLevel
	case "warn", "warning":
		logLevel = logrus.WarnLevel
	case "error":
		logLevel = logrus.ErrorLevel
	case "fatal":
		logLevel = logrus.FatalLevel
	case "":
		logLevel = logrus.InfoLevel
	default:
		fmt.Printf("Did not recognize log level: %s. Using INFO\n", requestedLevel)
		logLevel = logrus.InfoLevel
	}
	return logLevel
}
