package hub_registry

import (
	"encoding/json"
	"errors"
	"fmt"
	"github.com/jonboulle/clockwork"
	"io"
	"strconv"
	"time"
)

// Object representing a Hub
type Hub struct {
	ID          string     `json:"id,omitempty" valid:"required"`
	IP          string     `valid:"required,ipv4" faker:"ipv4"`
	Port        string     `valid:"required,port" faker:"customPort"`
	ClusterName string     `json:"clusterName"`
	Hostname    string     `valid:"host" faker:"domain_name"`
	Paused      bool       `json:"paused" faker:"-"`
	Healthy     bool       `json:"healthy" faker:"-"`
	SlotCounts  SlotCounts `json:"SlotCounts,omitempty"`
	CreatedAt   time.Time  `json:"created_at"` // TODO: Disallow ext modification
	UpdatedAt   time.Time  `json:"updated_at"` // TODO: Disallow ext modification
}

// Object representing the free and total slots in a hub
type SlotCounts struct {
	Free  int
	Total int
}

// Creates a new hub
// data The JSON Data to create the hub with
// clock The clock to utilize for created at times
func NewHub(data io.ReadCloser, clock clockwork.Clock) (*Hub, error) {
	hub := &Hub{
		Healthy: true, // default to healthy
		Paused: false, // default to unpaused
	}

	err := json.NewDecoder(data).Decode(hub)
	if err != nil {
		return nil, err
	}

	hub.CreatedAt = clock.Now()
	hub.UpdatedAt = clock.Now()
	return hub, err
}

// Initializes a Hub from a Redis Map
func NewHubFromMap(data map[string]string) (*Hub, error) {
	hub := &Hub{}

	// Convert the information from string to their proper datatype
	freeSlots, err := strconv.Atoi(data["slotcounts:free"])
	if err != nil {
		return hub, errors.New(fmt.Sprintf("problem parsing free slots from: %s", data["slotcounts:free"]))
	}
	totalSlots, err := strconv.Atoi(data["slotcounts:total"])
	if err != nil {
		return hub, errors.New(fmt.Sprintf("problem parsing total slots from: %s", data["slotcounts:total"]))
	}
	createdAt, err := time.Parse(time.RFC3339, data["created:at"])
	if err != nil {
		return hub, errors.New(fmt.Sprintf("problem parsing time from createdAt: %s", data["created:at"]))
	}
	updatedAt, err := time.Parse(time.RFC3339, data["updated:at"])
	if err != nil {
		return hub, errors.New(fmt.Sprintf("problem parsing time from updatedAt: %s", data["updated:at"]))
	}
	paused, err := strconv.ParseBool(data["paused"])
	if err != nil {
		return hub, errors.New(fmt.Sprintf("problem parsing paused: %s", data["paused"]))
	}
	healthy, err := strconv.ParseBool(data["healthy"])
	if err != nil {
		return hub, errors.New(fmt.Sprintf("problem parsing healthy: %s", data["healthy"]))
	}

	// Populate the struct
	hub.ID          = data["id"]
	hub.IP          = data["ip"]
	hub.Port        = data["port"]
	hub.ClusterName = data["clusterName"]
	hub.Hostname    = data["hostname"]
	hub.Paused      = paused
	hub.Healthy     = healthy
	hub.CreatedAt   = createdAt
	hub.UpdatedAt   = updatedAt
	hub.SlotCounts.Free  = freeSlots
	hub.SlotCounts.Total = totalSlots

	return hub, nil
}

// Returns boolean of if the Hub is accepting new sessions
func (hub *Hub) AcceptingNewSessions() bool {
	return hub.Healthy && !hub.Paused && hub.SlotCounts.Free > 0
}

// Returns boolean of if the Hub is drained, meaning there are no active sessions running on it
func (hub *Hub) Drained() bool {
	return hub.SlotCounts.Free == hub.SlotCounts.Total
}

// Merges an existing hub and json data
// existingHub The hub object that currently exists in the database
// data The JSON Data to update the hub with
func MergeHub(existingHub Hub, data io.ReadCloser, clock clockwork.Clock) (*Hub, error) {
	origCreatedAt := existingHub.CreatedAt
	err := json.NewDecoder(data).Decode(&existingHub)
	existingHub.CreatedAt = origCreatedAt
	existingHub.UpdatedAt = clock.Now()
	return &existingHub, err
}

// Serializes a hub to a map for redis
func (hub *Hub) ToRedisMap() map[string]interface{} {
	return map[string]interface{}{
		"id": hub.ID,
		"ip": hub.IP,
		"port": hub.Port,
		"clusterName": hub.ClusterName,
		"hostname": hub.Hostname,
		"paused": hub.Paused,
		"healthy": hub.Healthy,
		"slotcounts:free": hub.SlotCounts.Free,
		"slotcounts:total": hub.SlotCounts.Total,
		"created:at": hub.CreatedAt.Format(time.RFC3339),
		"updated:at": hub.UpdatedAt.Format(time.RFC3339),
	}
}
