// This package is a wrapper around CloudWatch Metrics that helps initialize the sessions,
// as well as get things like InstanceID for Dimensions
package instrumentor

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/autoscaling"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/sirupsen/logrus"
)

type Instrumentor struct {
	MetricClient           MetricWriter
	InstanceMetadataClient InstanceMetadataFetcher // Gathers things about the EC2 Environment
	Session                *session.Session
	InstanceID             string
	Region                 string
	AutoScalingGroupName   string
	Logger                 *logrus.Logger
}

type MetricWriter interface {
	PutMetricData(input *cloudwatch.PutMetricDataInput) (*cloudwatch.PutMetricDataOutput, error)
}

type InstanceMetadataFetcher interface {
	Available() bool
	Region() (string, error)
	GetInstanceIdentityDocument() (ec2metadata.EC2InstanceIdentityDocument, error)
}

// Creates a new Instrumentor Object
func New(logger *logrus.Logger) (*Instrumentor, error) {
	instrum := Instrumentor{
		Logger: logger,
	}
	err := instrum.Init()
	return &instrum, err
}

// Initializes an Instrumentor Object with default values
func (instrumentor *Instrumentor) Init() error {
	// If a Session wasn't provided, create one
	if instrumentor.Session == nil {
		awsSession, err := session.NewSession(aws.NewConfig())
		if err != nil {
			return err
		}
		instrumentor.Session = awsSession
	}

	// If an EC2 Metadata Client wasn't provided, create one
	if instrumentor.InstanceMetadataClient == nil {
		instrumentor.InstanceMetadataClient = ec2metadata.New(instrumentor.Session)
	}

	// If running in EC2, populate with the data of the ec2 instance
	if instrumentor.InstanceMetadataClient.Available() {
		region, err := instrumentor.InstanceMetadataClient.Region() // Get the Region
		if err != nil {
			return err
		}
		instrumentor.Region = region // Set the region

		idDoc, err := instrumentor.InstanceMetadataClient.GetInstanceIdentityDocument() // Get the Instance ID
		if err != nil {
			return err
		}
		instrumentor.InstanceID = idDoc.InstanceID // Set the instance ID

		// Get the ASG
		asgSvc := autoscaling.New(instrumentor.Session, &aws.Config{
			Region: &instrumentor.Region,
		})
		asgOutput, err := asgSvc.DescribeAutoScalingInstances(&autoscaling.DescribeAutoScalingInstancesInput {
			InstanceIds: []*string{&instrumentor.InstanceID},
		})

		// Set the ASG Name to the instrumentor struct
		if err != nil || len(asgOutput.AutoScalingInstances) != 1 {
			instrumentor.Logger.Errorf("Error fetching AutoScaling Group. Error: %v", err)
		} else {
			instrumentor.AutoScalingGroupName = *asgOutput.AutoScalingInstances[0].AutoScalingGroupName
		}
	} else { // else, use some defaults
		instrumentor.Logger.Info("(Instrumentor)Init() Detected not running in AWS. Setting region to default, us-west-2")
		instrumentor.Region = "us-west-2"
		instrumentor.InstanceID = "unknown"
	}

	// Set the region as reported from EC2, and create the Cloudwatch Configuration
	instrumentor.Session.Config.Region = &instrumentor.Region

	// If a Cloudwatch Client wasn't provided, create one
	if instrumentor.MetricClient == nil {
		instrumentor.MetricClient = cloudwatch.New(instrumentor.Session)
	}

	return nil
}
