package main

import (
	"errors"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/autoscaling"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
)

type AWS struct {
	Clients AWSClients
	Data    AWSData
}

type AWSClients struct {
	Session     *session.Session
	Autoscaling *autoscaling.AutoScaling
	Cloudwatch  *cloudwatch.CloudWatch
	EC2Metadata *ec2metadata.EC2Metadata
}

type AWSData struct {
	EC2 EC2Data
	ASG ASGData
}

type EC2Data struct {
	Region     string
	InstanceID string
}

type ASGData struct {
	Name string
}

type InstanceMetadataFetcher interface {
	Available() bool
	Region() (string, error)
	GetInstanceIdentityDocument() (ec2metadata.EC2InstanceIdentityDocument, error)
}

func NewAWS() (*AWS, error) {
	awsSess, err := session.NewSession(aws.NewConfig())
	if err != nil {
		return nil, err
	}

	ec2MD := ec2metadata.New(awsSess)
	ec2Data, err := FetchEC2Data(ec2MD)
	if err != nil {
		return nil, err
	}

	// Set the Region for the Session based on what the EC2 Instance is Running In
	awsSess.Config.Region = aws.String(ec2Data.Region)
	awsConfig := &aws.Config{ // use this for other client creation
		Region: aws.String(ec2Data.Region),
	}

	asgClient := autoscaling.New(awsSess, awsConfig)

	asgName, err := FetchASGName(asgClient, ec2Data.InstanceID)
	if err != nil {
		return nil, err
	}

	return &AWS{
		Clients: AWSClients{
			Session:     awsSess,
			Autoscaling: asgClient,
			Cloudwatch:  cloudwatch.New(awsSess, awsConfig),
			EC2Metadata: ec2MD,
		},
		Data: AWSData{
			EC2: ec2Data,
			ASG: ASGData{Name: asgName},
		},
	}, nil
}

func FetchEC2Data(metadata InstanceMetadataFetcher) (EC2Data, error) {
	ec2data := EC2Data{}

	if metadata.Available() == false {
		return ec2data, nil
	} else {
		// Region
		region, err := metadata.Region()
		if err != nil {
			return ec2data, err
		}
		ec2data.Region = region

		// InstanceID
		idDoc, err := metadata.GetInstanceIdentityDocument()
		if err != nil {
			return ec2data, err
		}
		ec2data.InstanceID = idDoc.InstanceID
	}

	return ec2data, nil
}

func FetchASGName(asgClient *autoscaling.AutoScaling, instanceID string) (string, error) {
	asgOutput, err := asgClient.DescribeAutoScalingInstances(&autoscaling.DescribeAutoScalingInstancesInput {
		InstanceIds: []*string{aws.String(instanceID)},
	})
	if err != nil {
		return "", err
	}
	if len(asgOutput.AutoScalingInstances) != 1 {
		return "", fmt.Errorf("unable to find asg by name. results: %v", asgOutput.AutoScalingInstances)
	}
	if asgOutput.AutoScalingInstances[0].AutoScalingGroupName == nil {
		return "", errors.New("asg group name was nil")
	}

	return *asgOutput.AutoScalingInstances[0].AutoScalingGroupName, nil
}
