package main

import (
	mocks2 "code.justin.tv/qe/grid_router/src/mocks"
	"errors"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/stretchr/testify/assert"
	"testing"
)

func TestFetchEC2Data(t *testing.T) {
	t.Run("when not on ec2, returns empty", func (t *testing.T) {
		mockFetcher := &mocks2.InstanceMetadataFetcher{}
		mockFetcher.On("Available").Return(false).Once()
		resp, err := FetchEC2Data(mockFetcher)
		assert.NoError(t, err)
		assert.Empty(t, resp)
	})

	t.Run("when on ec2, returns data", func (t *testing.T) {
		mockFetcher := &mocks2.InstanceMetadataFetcher{}
		testRegion := "testRegion"
		testInstanceID := "i-1234"

		mockInstanceIDResp := ec2metadata.EC2InstanceIdentityDocument{
			InstanceID: testInstanceID,
		}

		mockFetcher.On("Available").Return(true).Once()
		mockFetcher.On("Region").Return(testRegion, nil).Once()
		mockFetcher.On("GetInstanceIdentityDocument").Return(mockInstanceIDResp, nil).Once()

		resp, err := FetchEC2Data(mockFetcher)
		assert.NoError(t, err)
		assert.NotEmpty(t, resp)
		assert.Equal(t, testRegion, resp.Region)
		assert.Equal(t, testInstanceID, resp.InstanceID)
	})

	t.Run("returns errors", func (t *testing.T) {
		t.Run("region", func (t *testing.T) {
			mockFetcher := &mocks2.InstanceMetadataFetcher{}
			mockErr := "mock error for region"

			mockFetcher.On("Available").Return(true).Once()
			mockFetcher.On("Region").Return("", errors.New(mockErr)).Once()

			resp, err := FetchEC2Data(mockFetcher)
			assert.EqualError(t, err, mockErr)
			assert.Empty(t, resp)
		})

		t.Run("instance id", func (t *testing.T) {
			mockFetcher := &mocks2.InstanceMetadataFetcher{}
			mockErr := "mock error for id doc"

			mockFetcher.On("Available").Return(true).Once()
			mockFetcher.On("Region").Return("", nil).Once()
			mockFetcher.On("GetInstanceIdentityDocument").Return(
				ec2metadata.EC2InstanceIdentityDocument{}, errors.New(mockErr)).Once()

			resp, err := FetchEC2Data(mockFetcher)
			assert.EqualError(t, err, mockErr)
			assert.Empty(t, resp)
		})
	})
}
