// The entry point for the package
package main

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/healthcheck"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor"
	"code.justin.tv/qe/grid_router/src/pkg/session"
	"github.com/gorilla/handlers"
	"github.com/gorilla/mux"
	"github.com/sirupsen/logrus"
	"net/http"
	"time"
)

func main() {
	port := ":8080" // Port to run proxy on

	appConfig, err := config.New()
	if err != nil {
		logrus.Fatalf("Problem initializing app config. Error: %v", err) // use the global logger incase config had an error creating one
	}

	// Set up AWS Clients
	awsClients := &AWS{}
	if appConfig.LocalRun == false {
		awsClients, err = NewAWS()
		if err != nil {
			appConfig.Logger.Fatalf("Problem initializing AWS Clients. Err %v", err)
		}
	}

	// Set up Authentication
	secrets := &Secrets{}
	err = secrets.Init(appConfig.Logger, appConfig.LocalRun)
	if err != nil {
		appConfig.Logger.Fatalf("Problem initializing app secrets. Err: %v", err)
	}

	// Set up a Mux Router Object that's responsible to route requests to handlers
	router := mux.NewRouter()

	// Set up Metric Instrumentation
	instrumentClient := &instrumentor.Instrumentor{}
	if appConfig.LocalRun {
		instrumentClient = instrumentor.New(appConfig.Logger, nil, "")
		instrumentClient.DryRun = true
	} else {
		instrumentClient = instrumentor.New(appConfig.Logger, awsClients.Clients.Cloudwatch, awsClients.Data.ASG.Name)
		instrumentClient.DryRun = false
	}
	appConfig.Instrumentor = instrumentClient

	// Set up the Redis Client
	redis := Redis{}
	redis.Init(*secrets, appConfig.Logger)
	defer redis.Client.Close()

	// Create the Hub Registry
	registry := hub_registry.NewRegistry(instrumentClient, redis.Client, appConfig, time.Hour * 2)

	// Create the Hub API
	registryApi := hub_registry.NewAPI(registry, appConfig)

	// User friendly stdout
	appConfig.Logger.Infof("Server will run on http://localhost%s\n", port)
	appConfig.Logger.Infof("Press Ctrl C to exit.")

	headersOk := handlers.AllowedHeaders([]string{"X-Requested-With", "Content-Type", "Authorization"})
	originsOk := handlers.AllowedOrigins([]string{
		"http://localhost:3000",
		"http://dev.browsergrid.xarth.tv",
		"https://dev.browsergrid.xarth.tv",
		"http://browsergrid.xarth.tv",
		"https://browsergrid.xarth.tv",
	})
	methodsOk := handlers.AllowedMethods([]string{"GET", "HEAD", "POST", "PUT", "DELETE", "OPTIONS"})

	s := &http.Server{
		Addr:           ":8080",
		Handler:        handlers.CORS(originsOk, headersOk, methodsOk)(router),
		ReadTimeout:    305 * time.Second,
		WriteTimeout:   305 * time.Second,
	}

	/************
	Define Routes
	*************/
	router.HandleFunc("/cbg/api/hub/registry",      registryApi.GetHubs).Methods("GET")
	router.HandleFunc("/cbg/api/hub/registry/{id}", registryApi.GetHub).Methods("GET")
	router.HandleFunc("/cbg/api/hub/registry/{id}", registryApi.CreateHub).Methods("POST")
	router.HandleFunc("/cbg/api/hub/registry/{id}", registryApi.DeleteHub).Methods("DELETE")
	router.HandleFunc("/cbg/api/hub/registry/{id}/pause", registryApi.PauseHub).Methods("PUT")
	router.HandleFunc("/cbg/api/hub/registry/{id}/unpause", registryApi.UnpauseHub).Methods("PUT")

	// Healthcheck
	router.HandleFunc("/healthcheck", healthcheck.Handler) // Simple healthcheck route

	// Selenium Hub Session Requests
	sessionHandler := session.NewHandler(registryApi.Registry, secrets.APIKeys, appConfig)
	router.PathPrefix("/wd/hub/session").HandlerFunc(sessionHandler.Handle)

	/****************
	Set Up Async Jobs
	*****************/
	staleCleanupJob := hub_registry.NewStaleCleanUpJob(registryApi.Registry, appConfig)
	go staleCleanupJob.Start()

	// Launch HTTP Server
	if err := s.ListenAndServe(); err != nil {
		panic(err)
	}
}
