// This package stores Global App Configuration that multiple packages will need
package config

import (
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor"
	"fmt"
	"github.com/jonboulle/clockwork"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"os"
	"strings"
	"time"
)

const (
	logLevelEnvName = "LOG_LEVEL"
	defaultForwardRequestTimeout = time.Minute
	localRunENVName = "GR_LOCAL"
	defaultLocalRun = true
)

type Config struct {
	Logger         *logrus.Logger
	Clock          clockwork.Clock
	Instrumentor   *instrumentor.Instrumentor
	ForwardRequestTimeout time.Duration
	LocalRun       bool
}

// Creates a new configuration object
func New() (*Config, error) {
	config := &Config{
		Logger: logrus.New(),
		Clock:  clockwork.NewRealClock(),
		ForwardRequestTimeout: defaultForwardRequestTimeout,
		LocalRun: getLocalRun(),
	}
	config.Logger.SetLevel(getLogLevel())

	config.Logger.Infof("Local Run set to: %t", config.LocalRun)
	return config, nil
}

// Returns a config that should only be used in tests
func NewMock() *Config {
	logger, _ := test.NewNullLogger()
	return &Config{
		Logger: logger,
		Clock: clockwork.NewFakeClock(),
		ForwardRequestTimeout: defaultForwardRequestTimeout,
		LocalRun: true,
	}
}

// Returns a log level to use based on the environment variable logLevelEnvName
// No input will default to Info
// Unrecognized input will return Info
func getLogLevel() logrus.Level {
	var logLevel logrus.Level

	// toLower - case insensitive input
	switch requestedLevel := strings.ToLower(os.Getenv(logLevelEnvName)); requestedLevel {
	case "debug":
		logLevel = logrus.DebugLevel
	case "info":
		logLevel = logrus.InfoLevel
	case "warn", "warning":
		logLevel = logrus.WarnLevel
	case "error":
		logLevel = logrus.ErrorLevel
	case "fatal":
		logLevel = logrus.FatalLevel
	case "":
		logLevel = logrus.InfoLevel
	default:
		fmt.Printf("Did not recognize log level: %s. Using INFO\n", requestedLevel)
		logLevel = logrus.InfoLevel
	}
	return logLevel
}

// Fetch the configuration for if it's a local (non-prod) run
func getLocalRun() bool {
	// Local Run - whether the application is running locally or in an AWS Production Env
	localRun := strings.ToLower(os.Getenv(localRunENVName))
	if localRun == "false" {
		return false
	} else if localRun == "true" {
		return true
	} else {
		return defaultLocalRun
	}
}
