package config

import (
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"os"
	"reflect"
	"strings"
	"testing"
)

func TestNew(t *testing.T) {
	t.Run("automatically assigns", func (t *testing.T) {
		t.Run("a logger", func (t *testing.T) {
			config, err := New()
			assert.NoError(t, err)
			assert.NotNil(t, config)
			assert.NotNil(t, config.Logger)
			assert.IsType(t, &logrus.Logger{}, config.Logger)
		})

		t.Run("a real clock", func (t *testing.T) {
			config, err := New()
			assert.NoError(t, err)
			assert.NotNil(t, config)
			assert.NotNil(t, config.Clock)
			assert.Equal(t, reflect.TypeOf(config.Clock).String(), "*clockwork.realClock") // this is private, need to use string to access
		})

		t.Run("the default timeout", func (t *testing.T) {
			config, err := New()
			assert.NoError(t, err)
			assert.Equal(t, defaultForwardRequestTimeout, config.ForwardRequestTimeout)
		})
	})

	t.Run("LocalRun", func (t *testing.T) {
		t.Run("uses default when not provided", func (t *testing.T) {
			t.Run("the default Local Run flag", func (t *testing.T) {
				config, err := New()
				assert.NoError(t, err)
				assert.Equal(t, defaultLocalRun, config.LocalRun)
			})
		})

		t.Run("sets properly when provided", func (t *testing.T) {
			var localRunTests = []struct {
				in string
				out bool
			} {
				{"true", true},
				{"false", false},
				{"notvalid", defaultLocalRun},
				{"", defaultLocalRun},
			}

			for _, tt := range localRunTests {
				t.Run(tt.in, func(t *testing.T) {
					err := os.Setenv(localRunENVName, tt.in)
					require.NoError(t, err)
					defer os.Unsetenv(localRunENVName)
					
					config, err := New()
					require.NoError(t, err)
					assert.Equal(t, tt.out, config.LocalRun)
				})

				// case testing - case shouldn't matter
				t.Run(strings.ToUpper(tt.in), func(t *testing.T) {
					err := os.Setenv(localRunENVName, strings.ToUpper(tt.in))
					require.NoError(t, err)
					defer os.Unsetenv(localRunENVName)

					config, err := New()
					require.NoError(t, err)
					assert.Equal(t, tt.out, config.LocalRun)
				})
			}
		})
	})
}

func TestGetLogLevel(t *testing.T) {
	var levelTests = []struct {
		in string
		out logrus.Level
	} {
		{"debug", logrus.DebugLevel},
		{"info", logrus.InfoLevel},
		{"warn", logrus.WarnLevel},
		{"warning", logrus.WarnLevel},
		{"error", logrus.ErrorLevel},
		{"fatal", logrus.FatalLevel},
		{"", logrus.InfoLevel},
		{"notalevel", logrus.InfoLevel},
	}

	for _, tt := range levelTests {
		t.Run(tt.in, func(t *testing.T) {
			err := os.Setenv(logLevelEnvName, tt.in)
			assert.NoError(t, err)
			res := getLogLevel()
			assert.Equal(t, tt.out, res)
		})

		// case testing - case shouldn't matter
		t.Run(strings.ToUpper(tt.in), func(t *testing.T) {
			err := os.Setenv(logLevelEnvName, strings.ToUpper(tt.in))
			assert.NoError(t, err)
			res := getLogLevel()
			assert.Equal(t, tt.out, res)
		})
	}

	os.Unsetenv(logLevelEnvName) // Clean up :)
}
