package hub_registry

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/asaskevich/govalidator"
	"github.com/gorilla/mux"
	"net/http"
)

// API Methods pertaining to the Hub Registry
type Api struct {
	Registry  *RedisRegistry
	appConfig *config.Config
}

// Creates a new Hub API
func NewAPI(registry *RedisRegistry, appConfig *config.Config) Api {
	return Api{
		Registry: registry,
		appConfig: appConfig,
	}
}

// Object used for returning HTTP Errors from a method
type httpError struct {
	StatusCode int // The status code to return as
	Error error // The actual error message to be logged to server
	UserError string // What to display to the user
}

// GET /api/hub/registry
// Returns all of the hubs in the registry
func (api *Api) GetHubs(w http.ResponseWriter, r *http.Request) {
	hubs, err := api.Registry.GetHubs()
	if err != nil {
		api.appConfig.Logger.Errorf("When fetching hubs, encountered error: %v", err)
		http.Error(w, "unexpected error fetching hubs from database", http.StatusInternalServerError)
		return
	}
	api.writeObjectAsJsonOrError(w, hubs) // Write all hubs to output
}

// GET /api/hub/registry/:id
// Returns a specific hub, as specified by the param "id"
func (api *Api) GetHub(w http.ResponseWriter, r *http.Request) {
	// Fetch the ID, make sure it's valid
	hub, htErr := api.getHubFromRequest(r)
	if htErr != nil {
		api.appConfig.Logger.Errorf("When getting hub, encountered error %v", htErr.Error)
		http.Error(w, htErr.UserError, htErr.StatusCode)
		return
	}

	api.writeObjectAsJsonOrError(w, hub)
}

// POST /api/hub/registry/:id
// Creates or Updates a Hub.
// Note: PUT is not an accepted method as hubs will poll sending status. This method will automatically decide to create or update.
func (api *Api) CreateHub(w http.ResponseWriter, r *http.Request) {
	var httpErr *httpError // Store any error that might be returned

	// Fetch the ID, make sure it's valid
	id, httpErr := getHubIDFromRequest(r)
	if httpErr != nil {
		http.Error(w, httpErr.UserError, httpErr.StatusCode)
		return
	}

	// Open up the body of the request, which should contain the data
	body := r.Body
	defer body.Close()

	// Search if a hub already exists by that ID
	existingHub, err := api.Registry.GetHubById(id)
	if err != nil {
		api.appConfig.Logger.Errorf("createHub(): Encountered error looking up existing hub. Err: %v", err)
		http.Error(w, "Database Error", http.StatusInternalServerError)
		return
	}

	// Populate the Hub Object from Request Body
	hubToSave := &Hub{}
	if existingHub == nil { // if there is currently no new hub, create a new one
		hubToSave, err = NewHub(body, api.appConfig.Clock)
	} else { // if there is already a hub, merge the changes safely
		hubToSave, err = MergeHub(*existingHub, body, api.appConfig.Clock)
	}
	if err != nil {
		api.appConfig.Logger.Errorf("createHub(): Encountered error creating hub by body: %s", body)
		http.Error(w, "Problem decoding data. Ensure it is in json format.", http.StatusUnsupportedMediaType)
		return
	}

	// Assign the Hub ID incase it was not in the POST Data
	// This happens if the user did a call to /:id, so it stores in the params, but didn't provide it in the body
	hubToSave.ID = id

	// Validate the Struct
	validationResult, err := govalidator.ValidateStruct(hubToSave)
	if err != nil || validationResult == false {
		api.appConfig.Logger.Errorf("encountered an error validating: %v", err)
		http.Error(w, "Missing or invalid required field in body.", http.StatusUnprocessableEntity)
		return
	}

	err = api.Registry.SaveHub(hubToSave)
	if err != nil {
		api.appConfig.Logger.Errorf("CreateHub: Encountered error: %v", err)
		http.Error(w, "Database Error", http.StatusInternalServerError) // todo
		return
	}

	// Return all Hubs
	hubs, err := api.Registry.GetHubs()
	if err != nil {
		api.appConfig.Logger.Errorf("When fetching hubs, encountered error: %v", err)
		http.Error(w, "unexpected error fetching hubs", http.StatusInternalServerError)
		return
	}
	api.writeObjectAsJsonOrError(w, hubs)
}

// DELETE /api/hub/registry/:id
// Deletes a Hub
func (api *Api) DeleteHub(w http.ResponseWriter, r *http.Request) {
	// Fetch the ID, make sure it's valid
	id, httpErr := getHubIDFromRequest(r)
	if httpErr != nil {
		http.Error(w, httpErr.UserError, httpErr.StatusCode)
		return
	}

	// Make sure the hub exists
	exists, err := api.Registry.HubExists(id)
	if err != nil {
		http.Error(w, "Database Error", http.StatusInternalServerError)
		return
	}
	if !exists {
		http.Error(w, fmt.Sprintf("Could not find the specified hub id %s", id), http.StatusNotFound)
		return
	}

	// Delete the hub
	err = api.Registry.DeleteHub(id)
	if err != nil {
		// If there was an error, return back an http error
		api.appConfig.Logger.Errorf("While deleting hub, encountered error: %v", err)
		http.Error(w, "there was a problem deleting the hub", http.StatusNotFound)
		return
	} else {
		// Return all hubs
		hubs, err := api.Registry.GetHubs()
		if err != nil {
			api.appConfig.Logger.Errorf("When fetching hubs, encountered error: %v", err)
			http.Error(w, "unexpected error fetching hubs", http.StatusInternalServerError)
			return
		}
		api.writeObjectAsJsonOrError(w, hubs)
	}
}

// PUT /api/hub/registry/:id/pause
// Pauses a hub from accepting requests
func (api *Api) PauseHub(w http.ResponseWriter, r *http.Request) {
	// Fetch the ID, make sure it's valid
	hub, htErr := api.getHubFromRequest(r)
	if htErr != nil {
		api.appConfig.Logger.Errorf("When getting hub, encountered error %v", htErr.Error)
		http.Error(w, htErr.UserError, htErr.StatusCode)
		return
	}

	hub, err := api.Registry.PauseHub(hub)
	if err != nil {
		api.appConfig.Logger.Errorf("When pausing, encountered error: %v", err)
		http.Error(w, "Unexpected Error Pausing Hub", http.StatusInternalServerError)
		return
	}

	api.writeObjectAsJsonOrError(w, hub)
}

// PUT /api/hub/registry/:id/unpause
// Allows a hub to accept requests
func (api *Api) UnpauseHub(w http.ResponseWriter, r *http.Request) {
	// Fetch the ID, make sure it's valid
	hub, htErr := api.getHubFromRequest(r)
	if htErr != nil {
		api.appConfig.Logger.Errorf("When getting hub, encountered error %v", htErr.Error)
		http.Error(w, htErr.UserError, htErr.StatusCode)
		return
	}

	hub, err := api.Registry.UnpauseHub(hub)
	if err != nil {
		api.appConfig.Logger.Errorf("When unpausing, encountered error: %v", err)
		http.Error(w, "Unexpected Error Unpausing Hub", http.StatusInternalServerError)
		return
	}

	api.writeObjectAsJsonOrError(w, hub)
}

// Writes an object (usually a hub or an array of hubs) and encodes it as json
// Pass in the HTTP Writer to write to, and the object to write
func (api *Api) writeObjectAsJsonOrError(w http.ResponseWriter, obj interface{}) {
	err := json.NewEncoder(w).Encode(obj)
	if err != nil {
		api.appConfig.Logger.Errorf("writeObjectAsJsonOrError: Encountered error: %v", err)
		http.Error(w,"Unexpected Error", http.StatusInternalServerError)
		return
	}
}

// Returns a Hub based on an HTTP Request, which should have the ID in the request
// Returns an error if there was no hub id in the request, a problem reading from the database, or the hub did not exist
func (api *Api) getHubFromRequest(r *http.Request) (*Hub, *httpError) {
	id, htErr := getHubIDFromRequest(r)
	if htErr != nil {
		return nil, htErr
	}

	hub, err := api.Registry.GetHubById(id)
	if err != nil {
		return nil, &httpError{
			Error: err,
			UserError: "Unknown Error getting hub",
			StatusCode: 500,
		}
	}

	if hub == nil {
		return nil, &httpError{
			Error: errors.New("hub was nil from id " + id),
			UserError: "Hub not found",
			StatusCode: 404,
		}
	}

	return hub, nil
}

// Parses the Hub ID from the request parameters
// Takes an HTTP Request
// Returns the ID, and an HTTP Error object if the ID was not found
func getHubIDFromRequest(r *http.Request) (string, *httpError) {
	params := mux.Vars(r)

	// Fetch the ID, make sure it's valid
	id := params["id"]
	if id == "" {
		return "", &httpError{
			StatusCode: http.StatusUnprocessableEntity,
			Error: errors.New("missing the ID parameter"),
			UserError: "Missing the ID parameter",
		}
	}
	return id, nil
}
