package hub_registry_test

import (
	"bytes"
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	hub_registrytest "code.justin.tv/qe/grid_router/src/pkg/hub_registry/test"
	"encoding/json"
	"fmt"
	"github.com/alicebob/miniredis/v2"
	"github.com/go-redis/redis"
	"github.com/jonboulle/clockwork"
	"github.com/stretchr/testify/assert"
	"io/ioutil"
	"log"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"
)

func TestApi_GetHubs(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	reg := hub_registry.NewRegistry(nil, redisClient, config.NewMock(), time.Hour)
	api := createMockAPI(reg)
	/*
		When there are no hubs, should return an empty json
	*/
	reg.DBClient.FlushAll()
	w := httptest.NewRecorder()
	req := hub_registrytest.CreateTestRequestWithParams("", http.MethodGet, nil)
	api.GetHubs(w, req)
	resp := w.Result()

	assert.Equal(t, 200, resp.StatusCode)
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	assert.Equal(t, "[]\n", string(body[:]))

	/*
		When there are Hubs, it should return them
	*/
	mockHubs := hub_registrytest.LoadTestData(5, reg)

	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequestWithParams("", http.MethodGet, nil)
	api.GetHubs(w, req)
	resp = w.Result()

	assert.Equal(t, 200, resp.StatusCode)
	body, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}

	var hubsResult []*hub_registry.Hub
	err = json.Unmarshal(body, &hubsResult)
	assert.NoError(t, err)

	// Ensure the lengths match, and that each of the mockHubs is listed in the result
	assert.Equal(t, len(mockHubs), len(hubsResult))
	for _, hub := range mockHubs {
		hub_registrytest.AssertContainsHub(t, hubsResult, hub)
	}
}

func TestApi_GetHub(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	reg := hub_registry.NewRegistry(nil, redisClient, config.NewMock(), time.Hour)
	api := createMockAPI(reg)

	mockHubs := hub_registrytest.LoadTestData(5, reg)

	/*
		When Hub Exists
	*/
	testHub := mockHubs[0]

	w := httptest.NewRecorder()

	// Create the Request
	req := hub_registrytest.CreateTestRequestWithParams(testHub.ID, http.MethodGet, nil)

	// Call the function
	api.GetHub(w, req)
	resp := w.Result()
	assert.Equal(t, 200, resp.StatusCode)

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	assert.NoError(t, err)

	actualHub := &hub_registry.Hub{}
	err = json.Unmarshal(body, actualHub)
	hub_registrytest.AssertEqualHub(t, testHub, actualHub)

	/*
		When hub doesn't exist
	*/

	w = httptest.NewRecorder()
	id := "i-noexist"
	req = hub_registrytest.CreateTestRequestWithParams(id, http.MethodGet, nil)
	api.GetHub(w, req)
	resp = w.Result()
	assert.Equal(t, 404, resp.StatusCode)

	body, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	assert.NoError(t, err)
	assert.Equal(t, "Hub not found\n", string(body[:]))

	/*
		When no hub id was provided
	*/
	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequest("", http.MethodGet, nil)
	api.GetHub(w, req)
	resp = w.Result()
	assert.Equal(t, http.StatusUnprocessableEntity, resp.StatusCode)
	body, _ = ioutil.ReadAll(resp.Body)
	assert.Equal(t, "Missing the ID parameter\n", string(body[:]))
}

func TestApi_CreateHub(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	reg := hub_registry.NewRegistry(nil, redisClient, config.NewMock(), time.Hour)
	api := createMockAPI(reg)
	/*
		When there are no hubs
	*/

	testHub := hub_registrytest.CreateMockTestHub()
	resp := hub_registrytest.PostRequestToCreateHub(testHub, api)
	assert.Equal(t, 200, resp.StatusCode)

	// The hub we requested to add should now be in the db
	existRes, err := reg.HubExists(testHub.ID)
	assert.NoError(t, err)
	assert.True(t, existRes)

	// It should be not be paused by default
	hubFromDB, err := reg.GetHubById(testHub.ID)
	assert.NoError(t, err)
	assert.False(t, hubFromDB.Paused, "Hub should be paused by default")

	// It should be healthy by default
	assert.True(t, hubFromDB.Healthy, "Hub should be healthy by default")

	// Body should return all of the hubs
	respBody, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	var hubsResult []*hub_registry.Hub
	err = json.Unmarshal(respBody, &hubsResult)
	assert.NoError(t, err)
	hub_registrytest.AssertContainsHub(t, hubsResult, testHub)

	/*
		When there is already data
	*/
	mockHubs := hub_registrytest.LoadTestData(5, reg)
	testHub = hub_registrytest.CreateMockTestHub()
	resp = hub_registrytest.PostRequestToCreateHub(testHub, api)
	assert.Equal(t, 200, resp.StatusCode)

	// The hub we requested to add should now be in the db
	existRes, err = reg.HubExists(testHub.ID)
	assert.NoError(t, err)
	assert.True(t, existRes)

	// Body should return all of the hubs
	respBody, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	err = json.Unmarshal(respBody, &hubsResult)
	assert.NoError(t, err)
	hub_registrytest.AssertContainsHub(t, hubsResult, testHub)

	/*
		When specifying paused to true, it should be paused
	*/
	testHub = hub_registrytest.CreateMockTestHub()
	testHub.Paused = true
	resp = hub_registrytest.PostRequestToCreateHub(testHub, api)
	assert.Equal(t, 200, resp.StatusCode)
	// The hub we requested to add should now be in the db
	existRes, err = reg.HubExists(testHub.ID)
	assert.NoError(t, err)
	assert.True(t, existRes)

	hubFromDB, err = reg.GetHubById(testHub.ID)
	assert.NoError(t, err)
	assert.True(t, hubFromDB.Paused, "Hub should be paused")

	/*
		Healthy should be true by default
	*/
	newHub := hub_registrytest.CreateMockTestHub()
	assert.NotContains(t, hub_registrytest.GetHubs(t, reg), newHub, "The hub was already in the list of hubs")

	// Need to do things a bit more manual - createMockTestHub initializes Healthy to true
	// Create the recorder, take the object and marshall it into a json string
	w := httptest.NewRecorder()
	bodyData, err := json.Marshal(&newHub)
	if err != nil {
		log.Fatalf("problem marshaling data: %v", err)
	}

	// Remove the healthy attribute from the body string
	newBodyData := strings.Replace(string(bodyData), "\"healthy\":true,", "", 1)

	// Make the request
	req := hub_registrytest.CreateTestRequestWithParams(newHub.ID, http.MethodPost, bytes.NewReader([]byte(newBodyData)))
	api.CreateHub(w, req)
	resp = w.Result()

	// Make sure the response is okay
	assert.Equal(t, http.StatusOK, resp.StatusCode)
	hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, reg), newHub)

	// Make sure the healthy status did not change
	hubFromDB, err = reg.GetHubById(newHub.ID)
	assert.NoError(t, err)
	assert.True(t, hubFromDB.Healthy, "Healthy should default to true")

	/*
		When the data is an invalid type (i.e., sending a string when it should be json)
	*/
	beforeCount := len(hub_registrytest.GetHubs(t, reg))
	w = httptest.NewRecorder()
	bodyData = []byte("this data is invalid")
	id := "i-test"
	body := bytes.NewReader(bodyData)

	req = hub_registrytest.CreateTestRequestWithParams(id, http.MethodPost, body)

	api.CreateHub(w, req)
	resp = w.Result()
	assert.Equal(t, http.StatusUnsupportedMediaType, resp.StatusCode)
	assert.Equal(t, beforeCount, len(hub_registrytest.GetHubs(t, reg)), "the count of hubs should not have incremented")

	respBody, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}

	assert.Equal(t, "Problem decoding data. Ensure it is in json format.\n", string(respBody[:]))

	/*
		When there is no data
	*/
	beforeCount = len(hub_registrytest.GetHubs(t, reg))
	w = httptest.NewRecorder()
	bodyData = []byte("")
	id = "i-test"
	body = bytes.NewReader(bodyData)

	req = hub_registrytest.CreateTestRequestWithParams(id, http.MethodPost, body)

	api.CreateHub(w, req)
	resp = w.Result()
	assert.Equal(t, http.StatusUnsupportedMediaType, resp.StatusCode)
	assert.Equal(t, beforeCount, len(hub_registrytest.GetHubs(t, reg)), "the count of hubs should not have incremented")

	respBody, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}

	assert.Equal(t, "Problem decoding data. Ensure it is in json format.\n", string(respBody[:]))

	/*
		When parameter ID is empty
	*/
	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequest("", http.MethodPost, nil)
	api.CreateHub(w, req)
	resp = w.Result()
	assert.Equal(t, http.StatusUnprocessableEntity, resp.StatusCode)
	paramIdEmptyBody, _ := ioutil.ReadAll(resp.Body)
	assert.Equal(t, "Missing the ID parameter\n", string(paramIdEmptyBody[:]))

	/*
		Data Validation Tests
	*/

	var ipTest = []struct {
		in string
		out int
	}{
		{"", 422},
		{"notanip", 422},
		{"1234", 422},
		{"1.2.3.4", 200},
	}
	for index, tt := range ipTest {
		t.Logf("[%d] Testing in: %v out: %d", index, tt.in, tt.out)

		testHub = hub_registrytest.CreateMockTestHub()
		testHub.IP = tt.in
		resp = hub_registrytest.PostRequestToCreateHub(testHub, api)
		assert.Equal(t, tt.out, resp.StatusCode)
	}

	var portTest = []struct {
		in string
		out int
	}{
		{"", 422},
		{"notaport", 422},
		{"3.14", 422},
		{"4444", 200},
		{"80", 200},
	}
	for index, tt := range portTest {
		t.Logf("[%d] Testing in: %v out: %d", index, tt.in, tt.out)
		testHub = hub_registrytest.CreateMockTestHub()
		testHub.Port = tt.in
		resp = hub_registrytest.PostRequestToCreateHub(testHub, api)
		assert.Equal(t, tt.out, resp.StatusCode)
	}

	/*
		Update an existing hub
	*/
	hubToUpdate := mockHubs[0]
	newHub = hub_registrytest.CreateMockTestHub()
	newHub.ID = hubToUpdate.ID // assign the new hub as the same ID as the hub to update

	assert.NotContains(t, hub_registrytest.GetHubs(t, reg), newHub, "The hub was already in the list of hubs")
	resp = hub_registrytest.PostRequestToCreateHub(newHub, api)
	assert.Equal(t, http.StatusOK, resp.StatusCode)
	hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, reg), newHub)

	// Paused should remain defaulted to false
	hubFromDB, err = reg.GetHubById(newHub.ID)
	assert.NoError(t, err)
	assert.False(t, hubFromDB.Paused, "Paused should be defaulted to false")

	/*
		Update does not change pause attribute when pause is not specified
	*/

	// Preload it - enable paused
	hubToUpdate = mockHubs[0]
	hubToUpdate.Paused = true
	err = reg.SaveHub(hubToUpdate)
	assert.NoError(t, err)

	// Create the new hub
	newHub = hub_registrytest.CreateMockTestHub()
	newHub.ID = hubToUpdate.ID // assign the new hub as the same ID as the hub to update
	assert.NotContains(t, hub_registrytest.GetHubs(t, reg), newHub, "The hub was already in the list of hubs")

	// Need to do things a bit more manual - createMockTestHub initializes Pause to false
	// Create the recorder, take the object and marshall it into a json string
	w = httptest.NewRecorder()
	bodyData, err = json.Marshal(&newHub)
	if err != nil {
		log.Fatalf("problem marshaling data: %v", err)
	}

	// Remove the paused attribute from the body string
	newBodyData = strings.Replace(string(bodyData), "\"paused\":false,", "", 1)

	// Make the request
	req = hub_registrytest.CreateTestRequestWithParams(newHub.ID, http.MethodPost, bytes.NewReader([]byte(newBodyData)))
	api.CreateHub(w, req)
	resp = w.Result()

	// Make sure the response is okay
	assert.Equal(t, http.StatusOK, resp.StatusCode)
	hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, reg), newHub)

	// Make sure the paused status did not change
	hubFromDB, err = reg.GetHubById(newHub.ID)
	assert.NoError(t, err)
	assert.True(t, hubFromDB.Paused, "Paused should remain paused")

	/*
		Can update an existing hub to be paused
	*/
	hubToUpdate = mockHubs[0] // Fetch an existing hub to get its ID

	// Make sure it's not paused
	hubFromDB, err = reg.GetHubById(hubToUpdate.ID)
	assert.NoError(t, err)
	if hubFromDB.Paused { // error handle, set it back to unpaused
		hubFromDB.Paused = false
		err = reg.SaveHub(hubFromDB)
		assert.NoError(t, err)
	}

	assert.False(t, hubFromDB.Paused, "Precondition: the hub should be unpaused")

	// Update the contents
	newHub = hub_registrytest.CreateMockTestHub()
	newHub.ID = hubToUpdate.ID // assign the new hub as the same ID as the hub to update
	newHub.Paused = true // pause it

	resp = hub_registrytest.PostRequestToCreateHub(newHub, api)
	assert.Equal(t, http.StatusOK, resp.StatusCode)
	hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, reg), newHub)

	// Fetch hub from DB
	hubFromDB, err = reg.GetHubById(newHub.ID)
	assert.NoError(t, err)
	assert.True(t, hubFromDB.Paused, "Paused should have taken effect")

	/*
		Can update an existing hub to be unpaused
	*/
	hubToUpdate = mockHubs[0] // Fetch an existing hub to get its ID

	// Make sure it's still paused
	hubFromDB, err = reg.GetHubById(hubToUpdate.ID)
	assert.NoError(t, err)
	assert.True(t, hubFromDB.Paused, "Precondition: the hub should be paused")
	newHub = hub_registrytest.CreateMockTestHub()
	newHub.ID = hubToUpdate.ID // assign the new hub as the same ID as the hub to update
	newHub.Paused = false

	resp = hub_registrytest.PostRequestToCreateHub(newHub, api)
	assert.Equal(t, http.StatusOK, resp.StatusCode)
	hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, reg), newHub)

	// Fetch hub from DB
	hubFromDB, err = reg.GetHubById(newHub.ID)
	assert.NoError(t, err)
	assert.False(t, hubFromDB.Paused, "Unpause should have taken effect")

	/*
		Update an existing hub with bad data
	*/
	hubToUpdate = mockHubs[0]
	beforeCount = len(hub_registrytest.GetHubs(t, reg))
	w = httptest.NewRecorder()
	bodyData = []byte("this data is invalid")
	body = bytes.NewReader(bodyData)

	req = hub_registrytest.CreateTestRequestWithParams(hubToUpdate.ID, http.MethodPost, body)

	api.CreateHub(w, req)
	resp = w.Result()
	assert.Equal(t, http.StatusUnsupportedMediaType, resp.StatusCode)
	assert.Equal(t, beforeCount, len(hub_registrytest.GetHubs(t, reg)), "the count of hubs should not have incremented")

	/*
		Ensure UpdatedAt and CreatedAt change in time
	*/
	// Create a new fake clock and attach it to the registry
	clock := clockwork.NewFakeClock()
	reg = &hub_registry.RedisRegistry{
		DBClient:           redisClient,
		AppConfig:          config.NewMock(),
		HubConsideredStale: time.Hour,
	}
	reg.AppConfig.Clock = clock
	apiFake := createMockAPI(reg)

	// Create a Mock Hub and make a request to create it
	mockHub := hub_registrytest.CreateMockTestHub()
	expectedCreatedTime := clock.Now()
	expectedUpdatedTime := clock.Now()
	hub_registrytest.PostRequestToCreateHub(mockHub, apiFake)
	// Get the Hub from the registry, make sure times match
	actualHub := hub_registrytest.GetHubForAssert(hub_registrytest.GetHubs(t, reg), mockHub)
	assert.Equal(t, expectedCreatedTime, actualHub.CreatedAt)
	assert.Equal(t, expectedUpdatedTime, actualHub.UpdatedAt)

	// Move forward 5 minutes
	clock.Advance(time.Minute * 5)

	// Update ExpectedUpdateTime
	// Purposely not updating CreatedAt, as that shouldn't update
	expectedUpdatedTime = clock.Now()

	// Request the hub to be updated
	hub_registrytest.PostRequestToCreateHub(mockHub, apiFake)
	// Get the Hub from the registry, make sure times match
	actualHub = hub_registrytest.GetHubForAssert(hub_registrytest.GetHubs(t, reg), mockHub)
	assert.Equal(t, expectedCreatedTime, actualHub.CreatedAt)
	assert.Equal(t, expectedUpdatedTime, actualHub.UpdatedAt)
}

func TestApi_DeleteHub(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	reg := hub_registry.NewRegistry(nil, redisClient, config.NewMock(), time.Hour)
	api := createMockAPI(reg)
	mockHubs := hub_registrytest.LoadTestData(5, reg)
	/*
		When the hub exists
	*/

	hubToDelete := mockHubs[0]
	hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, reg), hubToDelete, "the hub should be included in the Hubs")

	w := httptest.NewRecorder()
	req := hub_registrytest.CreateTestRequestWithParams(hubToDelete.ID, http.MethodDelete, nil)

	api.DeleteHub(w, req)
	resp := w.Result()

	assert.Equal(t, 200, resp.StatusCode)
	hub_registrytest.AssertNotContainsHub(t, hub_registrytest.GetHubs(t, reg), hubToDelete, "the hub should be removed from the Hubs")

	// Body should return all of the hubs
	respBody, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	var hubsResult []*hub_registry.Hub
	err = json.Unmarshal(respBody, &hubsResult)
	assert.NoError(t, err)
	for _, hub := range hub_registrytest.GetHubs(t, reg) {
		hub_registrytest.AssertContainsHub(t, hubsResult, hub, "Response should include all of the current hubs")
	}

	/*
		When the hub does not exist
	*/

	id := "i-noexist"
	hubToDelete = &hub_registry.Hub{ID: id}
	hub_registrytest.AssertNotContainsHub(t, hub_registrytest.GetHubs(t, reg), hubToDelete, "The hub to delete should not exist")

	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequestWithParams(hubToDelete.ID, http.MethodDelete, nil)

	api.DeleteHub(w, req)
	resp = w.Result()

	assert.Equal(t, 404, resp.StatusCode)
	respBody, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}

	assert.Equal(t,
		fmt.Sprintf("Could not find the specified hub id %s\n", hubToDelete.ID),
		string(respBody[:]))

	/*
		When parameter ID is empty
	*/
	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequest("", http.MethodDelete, nil)
	api.DeleteHub(w, req)
	resp = w.Result()
	assert.Equal(t, http.StatusUnprocessableEntity, resp.StatusCode)
	paramIdEmptyBody, _ := ioutil.ReadAll(resp.Body)
	assert.Equal(t, "Missing the ID parameter\n", string(paramIdEmptyBody[:]))
}

func TestApi_PauseHub(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	reg := hub_registry.NewRegistry(nil, redisClient, config.NewMock(), time.Hour)
	api := createMockAPI(reg)

	// Create the test hub
	mockHubs := hub_registrytest.LoadTestData(5, reg)

	/*
	Pause should pause the hub
	 */
	testHub := mockHubs[0]
	assert.False(t,testHub.Paused)

	w := httptest.NewRecorder()
	req := hub_registrytest.CreateTestRequestWithParams(testHub.ID, http.MethodPut, nil)
	api.PauseHub(w, req)
	assert.Equal(t, 200, w.Result().StatusCode)

	dbHub, err := reg.GetHubById(testHub.ID)
	assert.NoError(t, err)
	assert.True(t, dbHub.Paused)

	/*
	An already paused hub should stay paused on future calls to PauseHub
	 */
	assert.True(t, dbHub.Paused)
	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequestWithParams(testHub.ID, http.MethodPut, nil)
	api.PauseHub(w, req)
	assert.Equal(t, 200, w.Result().StatusCode)

	dbHub, err = reg.GetHubById(testHub.ID)
	assert.NoError(t, err)
	assert.True(t, dbHub.Paused)

	/*
	Should return error if hub does not exist
	 */
	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequestWithParams("noexist", http.MethodPut, nil)
	api.PauseHub(w, req)
	assert.Equal(t, 404, w.Result().StatusCode)
	body, err := ioutil.ReadAll(w.Result().Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	assert.NoError(t, err)
	assert.Equal(t, "Hub not found\n", string(body[:]))
}

func TestApi_UnpauseHub(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	reg := hub_registry.NewRegistry(nil, redisClient, config.NewMock(), time.Hour)
	api := createMockAPI(reg)

	// Create the test hub
	mockHub := hub_registrytest.CreateMockTestHub()
	mockHub.Paused = true
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)

	/*
		Unpause should unpause the hub
	*/
	assert.True(t,mockHub.Paused)

	w := httptest.NewRecorder()
	req := hub_registrytest.CreateTestRequestWithParams(mockHub.ID, http.MethodPut, nil)
	api.UnpauseHub(w, req)
	assert.Equal(t, 200, w.Result().StatusCode)

	dbHub, err := reg.GetHubById(mockHub.ID)
	assert.NoError(t, err)
	assert.False(t, dbHub.Paused)

	/*
		An already paused hub should stay unpaused on future calls to unpause
	*/
	assert.False(t, dbHub.Paused)
	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequestWithParams(dbHub.ID, http.MethodPut, nil)
	api.UnpauseHub(w, req)
	assert.Equal(t, 200, w.Result().StatusCode)

	dbHub, err = reg.GetHubById(dbHub.ID)
	assert.NoError(t, err)
	assert.False(t, dbHub.Paused)

	/*
		Should return error if hub does not exist
	*/
	w = httptest.NewRecorder()
	req = hub_registrytest.CreateTestRequestWithParams("noexist", http.MethodPut, nil)
	api.UnpauseHub(w, req)
	assert.Equal(t, 404, w.Result().StatusCode)
	body, err := ioutil.ReadAll(w.Result().Body)
	if err != nil {
		t.Fatalf("error reading body: %v", err)
	}
	assert.NoError(t, err)
	assert.Equal(t, "Hub not found\n", string(body[:]))
}

func createMockAPI(registry *hub_registry.RedisRegistry) hub_registry.Api {
	return hub_registry.NewAPI(registry, registry.AppConfig)
}
