package hub_registry_test

import (
	"code.justin.tv/qe/grid_router/src/pkg/helpers"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	hub_registrytest "code.justin.tv/qe/grid_router/src/pkg/hub_registry/test"
	"encoding/json"
	"fmt"
	"github.com/jonboulle/clockwork"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"io/ioutil"
	"strconv"
	"strings"
	"testing"
	"time"
)

func TestNewHub(t *testing.T) {
	mockClock := clockwork.NewFakeClock()

	t.Run("returns a proper hub object", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()

		mockHub.CreatedAt = mockClock.Now()
		mockHub.UpdatedAt = mockClock.Now()

		jsonData := hub_registrytest.GetHubJsonReader(mockHub, "", "", "", "")

		result, err := hub_registry.NewHub(jsonData, mockClock)
		assert.NoError(t, err)
		hub_registrytest.AssertEqualHub(t, mockHub, result)
	})

	t.Run("paused", func (t *testing.T) {
		t.Run("defaults to false", func (t *testing.T) {
			mockHub  := hub_registrytest.CreateMockTestHub()
			jsonData := hub_registrytest.GetHubJsonReader(mockHub, "", "", "", "")
			result, err := hub_registry.NewHub(jsonData, mockClock)
			assert.NoError(t, err)
			require.NotNil(t, result)
			assert.False(t, result.Paused)
		})

		t.Run("can be supplied", func (t *testing.T) {
			mockHub  := hub_registrytest.CreateMockTestHub()
			jsonData := hub_registrytest.GetHubJsonReader(mockHub, "", "", "true", "")
			result, err := hub_registry.NewHub(jsonData, mockClock)
			assert.NoError(t, err)
			require.NotNil(t, result)
			assert.True(t, result.Paused)
		})
	})

	t.Run("healthy", func (t *testing.T) {
		t.Run("defaults to true", func (t *testing.T) {
			mockHub  := hub_registrytest.CreateMockTestHub()
			jsonData := hub_registrytest.GetHubJsonReader(mockHub, "", "", "", "")
			result, err := hub_registry.NewHub(jsonData, mockClock)
			assert.NoError(t, err)
			require.NotNil(t, result)
			assert.True(t, result.Healthy)
		})

		t.Run("can be supplied", func (t *testing.T) {
			mockHub  := hub_registrytest.CreateMockTestHub()
			jsonData := hub_registrytest.GetHubJsonReader(mockHub, "", "", "", "false")
			result, err := hub_registry.NewHub(jsonData, mockClock)
			assert.NoError(t, err)
			require.NotNil(t, result)
			assert.False(t, result.Healthy)
		})
	})

	t.Run("created_at", func (t *testing.T) {
		t.Run("defaults to current time", func (t *testing.T) {
			mockHub  := hub_registrytest.CreateMockTestHub()
			jsonData := hub_registrytest.GetHubJsonReader(mockHub, "", "", "", "")
			result, err := hub_registry.NewHub(jsonData, mockClock)
			assert.NoError(t, err)
			require.NotNil(t, result)
			assert.Equal(t, mockClock.Now(), result.CreatedAt)
		})

		t.Run("cannot be supplied", func (t *testing.T) {
			mockHub := hub_registrytest.CreateMockTestHub()

			createdAt := mockClock.Now().Add(time.Minute) // specify a time a minute into the future
			createdAtJson, err := json.Marshal(createdAt)
			assert.NoError(t, err)

			jsonData := hub_registrytest.GetHubJsonReader(mockHub, string(createdAtJson), "", "", "")
			result, err := hub_registry.NewHub(jsonData, mockClock)
			assert.NoError(t, err)
			require.NotNil(t, result)
			assert.NotEqual(t, createdAt, result.CreatedAt, "should not use the user inputted data")
			assert.Equal(t, mockClock.Now(), result.CreatedAt, "should use the time from the current clock")
		})
	})

	t.Run("updated_at", func (t *testing.T) {

		t.Run("cannot be supplied", func (t *testing.T) {
			mockHub := hub_registrytest.CreateMockTestHub()

			updatedAt := mockClock.Now().Add(time.Minute)
			updatedAtJson, err := json.Marshal(updatedAt)
			assert.NoError(t, err)

			jsonData := hub_registrytest.GetHubJsonReader(mockHub, "", string(updatedAtJson), "", "")
			result, err := hub_registry.NewHub(jsonData, mockClock)
			assert.NoError(t, err)
			require.NotNil(t, result)
			assert.NotEqual(t, updatedAt, result.UpdatedAt)
		})
	})

	t.Run("raises error on invalid json", func (t *testing.T) {
		reader := ioutil.NopCloser(strings.NewReader("invalid json"))
		result, err := hub_registry.NewHub(reader, mockClock)
		assert.Error(t, err)
		assert.Nil(t, result)
	})
}

func TestMergeHub(t *testing.T) {
	t.Run("returns back the proper hub", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()
		hubJson := hub_registrytest.GetHubJsonReader(mockHub, "", "", "", "")

		result, err := hub_registry.MergeHub(*mockHub, hubJson, clockwork.NewRealClock())
		assert.NoError(t, err)
		hub_registrytest.AssertEqualHub(t, mockHub, result)
	})

	t.Run("retains the original created at", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()

		// Create a clock to set the time to
		mockClock := clockwork.NewFakeClock()
		createdAtTime := mockClock.Now()
		require.NotEqual(t, mockHub.CreatedAt, createdAtTime)

		// Convert the createdAt Time to JSON
		createdAtJson, err := json.Marshal(mockClock.Now())
		assert.NoError(t, err)

		// Supply the created at time within the json
		hubJson := hub_registrytest.GetHubJsonReader(mockHub, string(createdAtJson), "", "", "")

		// Pass in the json
		result, err := hub_registry.MergeHub(*mockHub, hubJson, clockwork.NewRealClock())
		assert.NoError(t, err)
		assert.NotEqual(t, createdAtJson, result.CreatedAt, "the updated createdAt should not be in the result")
		assert.Equal(t, mockHub.CreatedAt, result.CreatedAt, "the original createdAt should be in the result")
	})

	t.Run("raises error on invalid json", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()
		reader := ioutil.NopCloser(strings.NewReader("invalid json"))
		result, err := hub_registry.MergeHub(*mockHub, reader, clockwork.NewRealClock())
		assert.Error(t, err)
		hub_registrytest.AssertEqualHub(t, mockHub, result) // unmodified hub should still be returned
	})
}

func TestNewHubFromMap(t *testing.T) {
	freeSlots := 5
	totalSlots := 10

	var initialValidMap = map[string]string {
		"id": "i-12345",
		"ip": "1.2.3.4",
		"port": "5678",
		"clusterName": "dev-t3st1ng",
		"hostname": "http://test.com",
		"paused": "false",
		"healthy": "true",
		"slotcounts:free": strconv.Itoa(freeSlots),
		"slotcounts:total": strconv.Itoa(totalSlots),
		"created:at": time.Now().Format(time.RFC3339),
		"updated:at": time.Now().Add(time.Minute).Format(time.RFC3339),
	}

	t.Run("parses all valid values", func (t *testing.T) {
		newMap := initialValidMap
		result, err := hub_registry.NewHubFromMap(newMap)
		require.NoError(t, err)

		assert.Equal(t, initialValidMap["id"], result.ID)
		assert.Equal(t, initialValidMap["ip"], result.IP)
		assert.Equal(t, initialValidMap["port"], result.Port)
		assert.Equal(t, initialValidMap["clusterName"], result.ClusterName)
		assert.Equal(t, initialValidMap["hostname"], result.Hostname)
		assert.False(t, result.Paused)
		assert.True(t, result.Healthy)
		assert.Equal(t, freeSlots, result.SlotCounts.Free)
		assert.Equal(t, totalSlots, result.SlotCounts.Total)

		// Assert the times in their proper RFC Format
		assert.Equal(t, initialValidMap["created:at"], result.CreatedAt.Format(time.RFC3339))
		assert.Equal(t, initialValidMap["updated:at"], result.UpdatedAt.Format(time.RFC3339))
	})

	t.Run("slotcounts", func (t *testing.T) {
		t.Run("when invalid free", func (t *testing.T) {
			newMap := helpers.CopyMap(&initialValidMap)
			newMap["slotcounts:free"] = "notanumber"

			res, err := hub_registry.NewHubFromMap(newMap)
			assert.EqualError(t, err, "problem parsing free slots from: notanumber")
			assert.Empty(t, res)
		})

		t.Run("when invalid total", func (t *testing.T) {
			newMap := helpers.CopyMap(&initialValidMap)
			newMap["slotcounts:total"] = "notanumber"

			res, err := hub_registry.NewHubFromMap(newMap)
			assert.EqualError(t, err, "problem parsing total slots from: notanumber")
			assert.Empty(t, res)
		})
	})

	t.Run("created at", func (t *testing.T) {
		newMap := helpers.CopyMap(&initialValidMap)
		newMap["created:at"] = "invalid time"

		res, err := hub_registry.NewHubFromMap(newMap)
		assert.EqualError(t, err, "problem parsing time from createdAt: invalid time")
		assert.Empty(t, res)
	})

	t.Run("updated at", func (t *testing.T) {
		newMap := helpers.CopyMap(&initialValidMap)
		newMap["updated:at"] = "invalid time"

		res, err := hub_registry.NewHubFromMap(newMap)
		assert.EqualError(t, err, "problem parsing time from updatedAt: invalid time")
		assert.Empty(t, res)
	})

	t.Run("paused", func (t *testing.T) {
		newMap := helpers.CopyMap(&initialValidMap)
		newMap["paused"] = "invalid bool"

		res, err := hub_registry.NewHubFromMap(newMap)
		assert.EqualError(t, err, "problem parsing paused: invalid bool")
		assert.Empty(t, res)
	})

	t.Run("healthy", func (t *testing.T) {
		newMap := helpers.CopyMap(&initialValidMap)
		newMap["healthy"] = "invalid bool"

		res, err := hub_registry.NewHubFromMap(newMap)
		assert.EqualError(t, err, "problem parsing healthy: invalid bool")
		assert.Empty(t, res)
	})
}

func TestHubToRedisMap(t *testing.T) {
	t.Run("returns a valid map", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()

		result := mockHub.ToRedisMap()
		assert.Equal(t, mockHub.ID, result["id"])
		assert.Equal(t, mockHub.IP, result["ip"])
		assert.Equal(t, mockHub.Port, result["port"])
		assert.Equal(t, mockHub.ClusterName, result["clusterName"])
		assert.Equal(t, mockHub.Hostname, result["hostname"])
		assert.Equal(t, mockHub.Paused, result["paused"])
		assert.Equal(t, mockHub.Healthy, result["healthy"])
		assert.Equal(t, mockHub.SlotCounts.Free, result["slotcounts:free"])
		assert.Equal(t, mockHub.SlotCounts.Total, result["slotcounts:total"])
		assert.Equal(t, mockHub.CreatedAt.Format(time.RFC3339), result["created:at"])
		assert.Equal(t, mockHub.UpdatedAt.Format(time.RFC3339), result["updated:at"])
	})
}

func TestHub_AcceptingNewSessions(t *testing.T) {
	var tests = []struct {
		inHealthy bool
		inPaused bool
		inFreeSlots int
		out bool
	}{
		{true, false, 1, true},
		{false, false, 1, false},
		{true, true, 1, false},
		{true, false, 0, false},
	}

	for _, tt := range tests {
		t.Run(fmt.Sprintf("%+v", tt), func (t *testing.T) {
			hub := hub_registry.Hub{
				Healthy: tt.inHealthy,
				Paused: tt.inPaused,
				SlotCounts: hub_registry.SlotCounts{
					Free:  tt.inFreeSlots,
					Total: 100,
				},
			}

			assert.Equal(t, tt.out, hub.AcceptingNewSessions())
		})
	}
}
