package hub_registry

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"errors"
	"github.com/davecgh/go-spew/spew"
	"time"
)

// Responsible for cleaning up stale Hubs
type StaleCleanUpJob struct {
	Registry        *RedisRegistry
	ConsideredStale time.Duration // The amount of time until a Hub is considered stale with no update
	PollDuration 	time.Duration // The amount of time to poll
	appConfig       *config.Config
}

// Creates a new Stale Request Cleanup Job with defaults set
func NewStaleCleanUpJob(reg *RedisRegistry, appConfig *config.Config) *StaleCleanUpJob {
	obj := StaleCleanUpJob{
		Registry:        reg,
		ConsideredStale: time.Second * 15,
		PollDuration: 	 time.Second * 10,
		appConfig:       appConfig,
	}
	return &obj
}

// Starts the StaleCleanUpJob that will poll continuously and delete any stale hubs
func (c *StaleCleanUpJob) Start() error {
	if c.Registry == nil {
		return errors.New("start(): a stale hub registry was referenced")
	}

	// Start an infinite loop
	for {
		err := c.UpdateStaleHubs()
		if err != nil {
			c.appConfig.Logger.Errorf("start(): While checking hubs for cleanup, encountered error: %v", err)
		}

		// Sleep between the next poll
		c.appConfig.Clock.Sleep(c.PollDuration)
	}
}

// Deletes stale hubs from the registry
func (c *StaleCleanUpJob) UpdateStaleHubs() error {
	// Protect against a nil registry
	if c.Registry == nil {
		return errors.New("UpdateStaleHubs(): a stale hub registry was referenced")
	}

	// Search for all hubs
	hubs, err := c.Registry.GetHubs()
	if err != nil {
		return err
	}
	for _, hub := range hubs {
		isHealthy, err := c.IsHubHealthy(hub) // check if the hub is unhealthy
		if err != nil {
			c.appConfig.Logger.Warnf("Ignoring an error when looping through hubs. Hub: %v, Error: %v",
				spew.Sdump(hub), err)
			continue
		}

		// If the health state has changed, update it
		if hub.Healthy != isHealthy {
			c.appConfig.Logger.Infof("UpdateStaleHubs(): Marking Hub [%s] Healthy to: %t", hub.ID, isHealthy)
			hub.Healthy = isHealthy
			return c.Registry.SaveHub(hub)
		}
	}
	return nil
}

// Determines if a Hub is healthy, based on the LastUpdatedAt and if it has passed the ConsideredStale amount
func (c *StaleCleanUpJob) IsHubHealthy(hub *Hub) (bool, error) {
	if hub == nil {
		return false, errors.New("isHubHealthy(): encountered reference to a nil hub")
	}

	// Grab the current time from the registry
	currentTime := c.appConfig.Clock.Now()

	// Determine the difference between now and when the hub was last updated, in seconds
	diff := currentTime.Sub(hub.UpdatedAt).Seconds()

	// Determine if the difference is less than the amount at which it should be considered stale
	return diff < c.ConsideredStale.Seconds(), nil
}
