package hub_registry_test

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	hub_registrytest "code.justin.tv/qe/grid_router/src/pkg/hub_registry/test"
	"github.com/alicebob/miniredis/v2"
	"github.com/go-redis/redis"
	"github.com/jonboulle/clockwork"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"testing"
	"time"
)

func TestStaleCleanUpJob_shouldDelete(t *testing.T) {
	mockRegistry := hub_registry.RedisRegistry{
		AppConfig: config.NewMock(),
	}
	mockJob := hub_registry.NewStaleCleanUpJob(&mockRegistry, mockRegistry.AppConfig)

	mockHub := hub_registrytest.CreateMockTestHub()
	mockHub.CreatedAt = mockRegistry.AppConfig.Clock.Now()
	mockHub.UpdatedAt = mockRegistry.AppConfig.Clock.Now()

	// Test all of the different amounts of time that pass between when a Hub is considered stale
	// Starting immediately, 1 second before, at the exact time, and 1 second after
	var durationTest = []struct {
		duration time.Duration
		expected bool
	}{
		{time.Second * 0, true},
		{mockJob.ConsideredStale - (time.Second * 1), true},
		{mockJob.ConsideredStale, false},
		{mockJob.ConsideredStale + (time.Second * 1), false},
	}
	for index, tt := range durationTest {
		t.Logf("[%d] Testing in: %v out: %v", index, tt.duration, tt.expected)

		// Create a new Fake Clock to reset the time from the previous test
		mockClock := clockwork.NewFakeClock()
		mockRegistry.AppConfig.Clock = mockClock

		// Advance the time based on the time table duration
		mockClock.Advance(tt.duration)
		result, err := mockJob.IsHubHealthy(mockHub)
		assert.NoError(t, err)
		assert.Equal(t, tt.expected, result)
	}

	// Test for error
	result, err := mockJob.IsHubHealthy(nil)
	assert.Error(t, err)
	assert.EqualError(t, err, "isHubHealthy(): encountered reference to a nil hub")
	assert.Equal(t, false, result) // Should default to false
}

func TestStaleCleanUpJob_UpdateStaleHubs(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	mockClock := clockwork.NewFakeClock()
	mockRegistry := hub_registry.RedisRegistry{
		DBClient:  redisClient,
		AppConfig: config.NewMock(),
	}
	mockRegistry.AppConfig.Clock = mockClock
	mockJob := hub_registry.NewStaleCleanUpJob(&mockRegistry, mockRegistry.AppConfig)

	// Create a Mock Hub
	mockHub := hub_registrytest.CreateMockTestHub()
	mockHub.CreatedAt = mockRegistry.AppConfig.Clock.Now()
	mockHub.UpdatedAt = mockRegistry.AppConfig.Clock.Now()

	// Add it to the Hubs registry
	err = mockRegistry.SaveHub(mockHub)
	assert.NoError(t, err)
	hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, &mockRegistry), mockHub)

	t.Run("after no time has passed, Hub should remain healthy", func (t *testing.T) {
		err = mockJob.UpdateStaleHubs()
		assert.NoError(t, err)
		hub_registrytest.AssertContainsHub(t, hub_registrytest.GetHubs(t, &mockRegistry), mockHub)
		hub, err := mockJob.Registry.GetHubById(mockHub.ID)
		assert.NoError(t, err)
		assert.True(t, hub.Healthy)
	})

	t.Run("After the considered stale time has passed, Hub should be unhealthy", func (t *testing.T) {
		hub, err := mockJob.Registry.GetHubById(mockHub.ID)
		require.NoError(t, err)
		originalUpdatedAtTime := hub.UpdatedAt // store this so we can reference it later

		mockClock.Advance(mockJob.ConsideredStale + (time.Second * 1))
		err = mockJob.UpdateStaleHubs()
		assert.NoError(t, err)
		hub, err = mockJob.Registry.GetHubById(mockHub.ID)
		assert.NoError(t, err)
		assert.False(t, hub.Healthy, "The hub should have healthy = false, but wasn't")

		t.Run("updated at should have not been modified, preventing the next run from setting it back to healthy",
			func (t *testing.T) {
			assert.Equal(t, originalUpdatedAtTime, hub.UpdatedAt)
		})
	})

	t.Run("Running a second time after stale time has passed should keep Hub set to unhealthy", func (t *testing.T) {
		err = mockJob.UpdateStaleHubs()
		assert.NoError(t, err)

		hub, err := mockJob.Registry.GetHubById(mockHub.ID)
		assert.NoError(t, err)
		assert.False(t, hub.Healthy)
	})

	t.Run("The hub should be set back to healthy when updated at is set", func (t *testing.T) {
		hub, err := mockJob.Registry.GetHubById(mockHub.ID)

		// Make the hub healthy again by updating the UpdatedTime
		mockClock.Advance(time.Second)
		hub.UpdatedAt = mockRegistry.AppConfig.Clock.Now()
		err = mockRegistry.SaveHub(hub)
		assert.NoError(t, err)

		err = mockJob.UpdateStaleHubs() // then run the function to make sure healthy is set to true
		hub, err = mockJob.Registry.GetHubById(hub.ID)
		assert.NoError(t, err)
		assert.True(t, hub.Healthy, "The hub should be healthy, but wasn't")
	})

	t.Run("when a nil registry is present, returns an error", func (t *testing.T) {
		mockJob.Registry = nil
		err = mockJob.UpdateStaleHubs()
		assert.Error(t, err)
		assert.EqualError(t, err, "UpdateStaleHubs(): a stale hub registry was referenced")
	})
}
