package hub_registry

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/helpers"
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor"
	"errors"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/go-redis/redis"
	"time"
)

var redisNamespace = "registry:hub" // The namespace where hubs are stored within redis

type Registry interface {
	GetHubs() ([]*Hub, error)
	DeleteHub(id string) error
	GetHubById(id string) (*Hub, error)
	HubExists(id string) (bool, error)
	PauseHub(hub *Hub) (*Hub, error)
	UnpauseHub(hub *Hub) (*Hub, error)
	SaveHub(hub *Hub) error
	PollForAvailableHub(timeBetweenPolls time.Duration, timeout time.Duration) (*Hub, error)
	FindAvailableHub() (*Hub, error)
}

// Methods and Data pertaining to the Hub Registry
type RedisRegistry struct {
	Instrumentor       *instrumentor.Instrumentor
	DBClient           redis.Cmdable
	AppConfig          *config.Config
	HubConsideredStale time.Duration // at what point a hub should be considered stale
	LastHubIndex       int           // stores the last hub index searched by the registry
}

// Creates a new registry
func NewRegistry(insturmentor *instrumentor.Instrumentor, redisClient *redis.Client, appConfig *config.Config,
	hubConsideredStale time.Duration) *RedisRegistry {
	return &RedisRegistry{
		Instrumentor:       insturmentor,
		DBClient:           redisClient,
		AppConfig:          appConfig,
		HubConsideredStale: hubConsideredStale,
		LastHubIndex:       -1, // initialize as a number to indicate "never ran"
	}
}

// Returns Hubs from the Registry
func (reg *RedisRegistry) GetHubs() ([]*Hub, error) {
	var hubs = make([]*Hub, 0) // Initialize the array of the hubs
	if reg.DBClient == nil {
		return hubs, errors.New("no db client in registry")
	}

	// Gather all of the keys within the hub registry
	redisKeys, err := reg.getHubKeys()
	if err != nil {
		return hubs, err
	}

	// Now that we have the keys, loop through all of them and get the record for each
	// Set up a MULTI Pipeline to the Redis server for each key in one transaction
	pipe := reg.DBClient.TxPipeline()
	var commands []*redis.StringStringMapCmd
	for _, key := range redisKeys {
		res := pipe.HGetAll(key)
		commands = append(commands, res)
	}
	_, err = pipe.Exec() // Execute the Pipeline
	if err != nil {
		return hubs, err
	}

	// Loop through the results and create a hub object from them
	for _, command := range commands {
		res, err := command.Result()
		if err != nil {
			reg.AppConfig.Logger.Errorf("While fetching redis results, encountered err: %v", err)
			continue
		}

		hub, err := NewHubFromMap(res)
		if err != nil {
			reg.AppConfig.Logger.Errorf("Problem unmarshalling hub from redis. Err: %v", err)
			continue
		}

		hubs = append(hubs, hub)
	}

	return hubs, nil
}

// Goes through all Hubs and deletes the hub based on the ID
// Provide the ID of a hub, case sensitive
// Returns an error if the hub could not be found
func (reg *RedisRegistry) DeleteHub(id string) error {
	res, err := reg.DBClient.Del(HubDBIdentifier(id)).Result()
	reg.AppConfig.Logger.Infof("Hub ID '%s' Deleted. Number of keys deleted from redis: %v", id, res)
	return err
}

// Provide the ID of the Hub to find
// Returns a Hub from the list of stored hubs
func (reg *RedisRegistry) GetHubById(id string) (*Hub, error) {
	return reg.getHubFromRedis(HubDBIdentifier(id))
}

// Returns a boolean on if the hub exists in the database
func (reg *RedisRegistry) HubExists(id string) (bool, error) {
	res, err := reg.DBClient.Exists(HubDBIdentifier(id)).Result()
	if err != nil {
		reg.AppConfig.Logger.Errorf("While checking if hub exists, encountered err: %v", err)
		return false, err
	}
	return res == int64(1), err // returns 1 if exists
}

// Pauses a Hub given its ID
func (reg *RedisRegistry) PauseHub(hub *Hub) (*Hub, error) {
	return reg.updatePauseAttribute(hub, true)
}

// Unpauses a Hub given its ID
func (reg *RedisRegistry) UnpauseHub(hub *Hub) (*Hub, error) {
	return reg.updatePauseAttribute(hub, false)
}

// Updates the pause attribute
// Provide a hub to make the update against
// Provide a bool of whether to pause or unpause
func (reg *RedisRegistry) updatePauseAttribute(hub *Hub, paused bool) (*Hub, error) {
	if hub == nil {
		return nil, errors.New("received a nil hub")
	}

	hub.Paused = paused
	err := reg.SaveHub(hub)
	return hub, err
}

// Saves the Hub to the Redis DB
func (reg *RedisRegistry) SaveHub(hub *Hub) error {
	if reg.DBClient == nil {
		return errors.New("no Redis Client associated with registry")
	}
	if hub == nil {
		return errors.New("nil hub passed to method")
	}

	// Create a pipe that will set the redis key, and then set the expiration to 2 hours (to prevent stale hubs)
	// if a new update comes into this method, it will reset the expiration
	redisKey := HubDBIdentifier(hub.ID)
	pipe := reg.DBClient.TxPipeline()
	_, err := pipe.HMSet(redisKey, hub.ToRedisMap()).Result()
	if err != nil {
		return err
	}

	// If the registry was initialized without a value, it will be 0. Don't expire for 0, that'd be removed immediately.
	if reg.HubConsideredStale > 0 {
		pipe.Expire(redisKey, reg.HubConsideredStale)
	}

	_, err = pipe.Exec()
	return err
}

// Gets a Hub from Redis
func (reg *RedisRegistry) getHubFromRedis(redisID string) (*Hub, error) {
	if reg.DBClient == nil {
		return nil, errors.New("no Redis Client associated with registry")
	}

	// Pull the hash from Redis
	res, err := reg.DBClient.HGetAll(redisID).Result()
	if err != nil {
		return nil, err
	}

	if err == redis.Nil || len(res) == 0 { // If the result was blank (key didn't exist)
		return nil, nil
	}

	// Create a Hub object and initialize it with the data
	return NewHubFromMap(res)
}

// Returns all Hub keys from the Redis Database
func (reg *RedisRegistry) getHubKeys() ([]string, error) {
	if reg.DBClient == nil {
		return nil, errors.New("no db in registry")
	}
	// Gather all of the keys within the hub registry
	var cursor uint64
	var redisKeys []string
	// Scan the DB for any keys (wildcard *) within the namespace
	// Response is limited to a length of count, so keep scanning (for loop) until finished
	for {
		keys, cursor, err := reg.DBClient.Scan(cursor, fmt.Sprintf("%s:*", redisNamespace), 50).Result()
		if err != nil {
			return nil, err
		}

		// Take all of the keys from this scan and add it to the array
		redisKeys = append(redisKeys, keys...)

		// When the cursor is back to 0, no more results
		if cursor == 0 {
			break
		}
	}

	// Ensure the redis keys are unique - Scan could return duplicates
	return helpers.UniqueSlice(redisKeys), nil
}

// Returns an available Hub from the registry, and will keep polling if one is not available
// Provide a timeBetweenPolls amount for how often to check the registry for an available hub
// Provide a timeout, and this function will stop polling after the timeout has been reached.
func (reg *RedisRegistry) PollForAvailableHub(timeBetweenPolls time.Duration, timeout time.Duration) (*Hub, error) {
	// Determine the time at which it should time out
	endBy := reg.AppConfig.Clock.Now().Add(timeout)
	startTime := reg.AppConfig.Clock.Now()
	var hub *Hub
	var err error

	// Loop until the timeout has been reached
	for reg.AppConfig.Clock.Now().Before(endBy) {
		hub, err = reg.FindAvailableHub()
		if err == nil && hub != nil { // If found a hub, break from loop. Else, retry
			break
		}
		if err != nil {
			reg.AppConfig.Logger.Warnf("Ignoring error while attempting to find available hub: %v", err)
		}

		time.Sleep(timeBetweenPolls)
	}

	// Report the duration to metrics (as a goroutine - async, get back to the client faster)
	go func() {
		err := reg.instrumentDurationToFindHub(reg.AppConfig.Clock.Now().Sub(startTime))
		if err != nil {
			reg.AppConfig.Logger.Errorf("instrumentDurationToFindHub(): Encountered error: %v", err)
		}
	}()

	return hub, err
}

// Returns an available Hub from the registry
/*
Distribution Algorithm Notes:
If more than 1 hub is registered, Grid Router will start scanning from where it last left off
if it hits the end of the array, and can't find capacity it will look at the beginning
Decided not to simply sort array by free slots. It would be the easiest, but might be expensive as we often
  happen onto a hub w/ free slots due to distributing
*/
func (reg *RedisRegistry) FindAvailableHub() (*Hub, error) {
	hubs, err := reg.GetHubs()
	if err != nil {
		return nil, err
	}

	// Initializes index at which to start searching. See method documentation for details
	startingIndex := 0

	// If more than one hub, and last index less than total hubs, increment the index
	if len(hubs) > 1 && reg.LastHubIndex < (len(hubs) - 1) {
		startingIndex = reg.LastHubIndex + 1
	}

	// Start scanning from where it left off
	for i := startingIndex; i < len(hubs); i++ {
		hub := hubs[i]
		if hub.AcceptingNewSessions() {
			reg.LastHubIndex = i
			return hub, nil
		}
	}

	// If started in the middle (not beginning), scan from the beginning to the point it scanned last
	if startingIndex != 0 {
		for i := 0; i < startingIndex; i++ {
			hub := hubs[i]
			if hub.AcceptingNewSessions() {
				reg.LastHubIndex = i
				return hub, nil
			}
		}
	}

	// If it got here, there was nothing available
	return nil, errors.New("no available hub")
}

// Sends a Metric to Cloudwatch on how long it took Grid Router to return an available hub
// This can help us understand how long a client is waiting to get a hub with capacity
// Takes a duration that it took to find
func (reg *RedisRegistry) instrumentDurationToFindHub(durationToFind time.Duration) error {
	if reg.Instrumentor == nil {
		return errors.New("no instrumentor client in registry")
	}

	// Get the Cloudwatch Metric Input from the Duration To Find
	input, err := reg.getMetricInputFromDurationToFindHub(durationToFind)
	if err != nil {
		return err
	}

	// Send the data to cloudwatch. Throw away result, don't need it
	reg.AppConfig.Logger.Debugf("instrumentDurationToFindHub() Sending Metric to Cloudwatch: %s", input.String())
	result, err := reg.Instrumentor.PutMetricData(input)
	reg.AppConfig.Logger.Debugf("instrumentDurationToFindHub() Cloudwatch Metric Result: %s", result.String())

	if err != nil {
		reg.AppConfig.Logger.Errorf("instrumentDurationToFindHub(): Encountered error: %v", err)
	}
	return err
}

// Takes a duration and converts it to Milliseconds as a Float
func (reg *RedisRegistry) convertDurationToFloatMS(value time.Duration) float64 {
	return float64(value.Nanoseconds() / (int64(time.Millisecond)/int64(time.Nanosecond)))
}

// Takes the duration to find an available hub and creates a Cloudwatch Metric Input for it
func (reg *RedisRegistry) getMetricInputFromDurationToFindHub(durationToFind time.Duration) (*cloudwatch.PutMetricDataInput, error) {
	// Error check necessary values - need an instance id for the dimensions
	if reg.Instrumentor == nil {
		return nil, errors.New("missing instrumentor")
	}
	if reg.Instrumentor.AutoScalingGroupName == "" {
		return nil, errors.New("missing asg")
	}

	// Convert Nanoseconds to Milliseconds, and then into a float64 for AWS
	value := reg.convertDurationToFloatMS(durationToFind)

	// Create the Metric Data Input Object
	return &cloudwatch.PutMetricDataInput{
		Namespace: aws.String("CBG"),
		MetricData: []*cloudwatch.MetricDatum{
			{
				MetricName: aws.String("DurationToFindHub"),
				Timestamp: aws.Time(reg.AppConfig.Clock.Now()),
				Value: &value,
				Unit: aws.String(cloudwatch.StandardUnitMilliseconds),
				Dimensions: []*cloudwatch.Dimension{
					{Name: aws.String("AutoScalingGroupName"), Value: aws.String(reg.Instrumentor.AutoScalingGroupName),},
				},
			},
		},
	}, nil
}

// Given a Hub ID, returns the identifier/key of how to reference that Hub within the Database
func HubDBIdentifier(hubID string) string {
	return fmt.Sprintf("%s:%s", redisNamespace, hubID)
}
