package hub_registry

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor"
	instrumentorMocks "code.justin.tv/qe/grid_router/src/pkg/instrumentor/mocks"
	"errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/stretchr/testify/assert"
	"testing"
	"time"
)

func TestRegistry_GetHubFromRedis(t *testing.T) {
	t.Run("returns an error if no dbclient", func (t *testing.T) {
		reg := RedisRegistry{}
		res, err := reg.getHubFromRedis("noMatter")
		assert.EqualError(t, err, "no Redis Client associated with registry")
		assert.Nil(t, res)
	})
}

func TestGetHubKeys(t *testing.T) {
	t.Run("returns an error if no db client", func (t *testing.T) {
		r := RedisRegistry{}
		res, err := r.getHubKeys()
		assert.EqualError(t, err, "no db in registry")
		assert.Nil(t, res)
	})
}

func TestRegistryconvertDurationToFloatMS(t *testing.T) {
	mockReg := RedisRegistry{
		AppConfig: config.NewMock(),
	}

	/*
		Ensure that durations convert to ms as expected
	*/
	var valTest = []struct {
		in time.Duration
		out float64
	}{
		{time.Second, float64(1000)},
		{time.Millisecond * 500, float64(500)},
		{time.Minute, float64(60000)},
	}
	for _, tt := range valTest {
		assert.Equal(t, tt.out, mockReg.convertDurationToFloatMS(tt.in), "expected %v to equal %v", tt.in, tt.out)
	}
}

func TestRegistrygetMetricInputFromDurationToFindHub(t *testing.T) {
	// Set up some expected values
	mockDurationToFind := time.Second // The mock value for how long it took to find the hub
	expectedNameSpace := "CBG"
	expectedMetricName := "DurationToFindHub"
	expectedASG        := "asg-123"

	/*
		Returns an error when the instrumentor object is missing
	*/
	mockReg := RedisRegistry{
		AppConfig: config.NewMock(),
	}
	result, err := mockReg.getMetricInputFromDurationToFindHub(mockDurationToFind)
	assert.EqualError(t, err, "missing instrumentor")

	/*
		Returns an error when the asg is missing
	*/
	mockInstrumentor := &instrumentor.Instrumentor{
		Logger: config.NewMock().Logger,
		MetricClient: &instrumentorMocks.MetricWriter{},
		AutoScalingGroupName: "",
		DryRun: false,
	}
	mockReg = RedisRegistry{
		Instrumentor: mockInstrumentor,
		AppConfig:    config.NewMock(),
	}
	result, err = mockReg.getMetricInputFromDurationToFindHub(mockDurationToFind)
	assert.EqualError(t, err, "missing asg")

	/*
		When all values provided, should contain all values
	*/
	mockInstrumentor = &instrumentor.Instrumentor{
		Logger: config.NewMock().Logger,
		MetricClient: nil,
		AutoScalingGroupName: expectedASG,
		DryRun: false,
	}
	mockReg = RedisRegistry{
		Instrumentor: mockInstrumentor,
		AppConfig:    config.NewMock(),
	}
	expectedMetricValue := mockReg.convertDurationToFloatMS(mockDurationToFind)
	currentTime := mockReg.AppConfig.Clock.Now()

	// Create the Expected Output Struct
	expectedOutput := &cloudwatch.PutMetricDataInput{
		Namespace: &expectedNameSpace,
		MetricData: []*cloudwatch.MetricDatum{
			{
				MetricName: &expectedMetricName,
				Timestamp: &currentTime,
				Value: &expectedMetricValue,
				Unit: aws.String(cloudwatch.StandardUnitMilliseconds),
				Dimensions: []*cloudwatch.Dimension{
					{Name: aws.String("AutoScalingGroupName"), Value: &expectedASG},
				},
			},
		},
	}

	// Make the call, ensure no errors and equals expected
	result, err = mockReg.getMetricInputFromDurationToFindHub(mockDurationToFind)
	assert.NoError(t, err)
	assert.Equal(t, expectedOutput, result)
}

func TestRegistryinstrumentDurationToFindHub(t *testing.T) {
	metricWriterExpectedCalls := 0 // Tracks how many times the Metric Writer method was called
	mockDurationToFind := time.Second // The mock value for how long it took to find the hub

	/*
		Returns an error if there is no instrumentor
	*/
	mockReg := RedisRegistry{
		AppConfig: config.NewMock(),
	}

	err := mockReg.instrumentDurationToFindHub(time.Second)
	assert.EqualError(t, err, "no instrumentor client in registry")

	/*
		When all values provided, it should make the call to Cloudwatch
	*/

	mockASGName := "asg-123"
	mockMetricWriter := &instrumentorMocks.MetricWriter{} // The mock Metric Writer
	mockInstrumentor := &instrumentor.Instrumentor{
		Logger: config.NewMock().Logger,
		MetricClient: mockMetricWriter,
		AutoScalingGroupName: mockASGName,
		DryRun: false,
	}
	mockReg.Instrumentor = mockInstrumentor

	// Gather the input data from another method, which is being tested elsewhere
	mockInput, err := mockReg.getMetricInputFromDurationToFindHub(mockDurationToFind)
	assert.NoError(t, err)
	assert.NotNil(t, mockInput)

	// Set up the mock to expect that value
	mockMetricWriter.On("PutMetricData", mockInput).Return(&cloudwatch.PutMetricDataOutput{}, nil).Once()

	// Make the call to instrument
	err = mockReg.instrumentDurationToFindHub(mockDurationToFind)
	metricWriterExpectedCalls += 1 // Increment number of calls in this test case
	mockMetricWriter.AssertNumberOfCalls(t, "PutMetricData", metricWriterExpectedCalls)
	assert.NoError(t, err)

	/*
		Test than an error gets sent back if Cloudwatch returns an error
	*/
	mockError := errors.New("this is an error")
	mockMetricWriter.On("PutMetricData", mockInput).Return(&cloudwatch.PutMetricDataOutput{}, mockError).Once()

	err = mockReg.instrumentDurationToFindHub(mockDurationToFind)
	metricWriterExpectedCalls += 1 // Increment number of calls in this test case
	mockMetricWriter.AssertNumberOfCalls(t, "PutMetricData", metricWriterExpectedCalls)
	assert.Equal(t, mockError, err)
}
