package hub_registry_test

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry/mocks"
	hub_registrytest "code.justin.tv/qe/grid_router/src/pkg/hub_registry/test"
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor"
	instrumentorMocks "code.justin.tv/qe/grid_router/src/pkg/instrumentor/mocks"
	"errors"
	"fmt"
	"github.com/alicebob/miniredis/v2"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/go-redis/redis"
	"github.com/jonboulle/clockwork"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"strconv"
	"testing"
	"time"
)

func TestRegistry_GetHubs(t *testing.T) {
	t.Run("returns an error if no db client", func (t *testing.T) {
		reg := hub_registry.RedisRegistry{}
		res, err := reg.GetHubs()
		assert.EqualError(t, err, "no db client in registry")
		assert.Empty(t, res)
	})

	t.Run("when there are no hubs", func (t *testing.T) {
		r := hub_registrytest.NewTestRedis()
		reg := hub_registry.RedisRegistry{
			DBClient: r,
		}

		hubs, err := reg.GetHubs()
		assert.NoError(t, err)
		assert.Empty(t, hubs)
	})

	t.Run("where there are one or more hubs", func (t *testing.T) {
		mockHub1 := hub_registrytest.CreateMockTestHub()
		mockHub2 := hub_registrytest.CreateMockTestHub()

		// Set up a test redis instance
		r := hub_registrytest.NewTestRedis()

		// Set up the registry
		reg := hub_registry.RedisRegistry{
			DBClient:  r,
			AppConfig: config.NewMock(),
		}

		// Save the hubs into the DB
		err := reg.SaveHub(mockHub1)
		require.NoError(t, err)
		err = reg.SaveHub(mockHub2)
		require.NoError(t, err)

		// Make the test calls
		hubs, err := reg.GetHubs()
		assert.NoError(t, err)
		assert.NotEmpty(t, hubs)
		hub_registrytest.AssertContainsHub(t, hubs, mockHub1)
		hub_registrytest.AssertContainsHub(t, hubs, mockHub2)
	})

	t.Run("when there is an error from redis", func (t *testing.T) {
		// Set up a test redis instance
		r := hub_registrytest.NewTestRedis()
		reg := hub_registry.RedisRegistry{
			DBClient:  r,
			AppConfig: config.NewMock(),
		}

		mockHub1 := hub_registrytest.CreateMockTestHub()
		mockHub1RedisKey := hub_registry.HubDBIdentifier(mockHub1.ID)
		hubMap1 := make(map[string]string)
		for key, value := range mockHub1.ToRedisMap() { // convert map[string]interface to map[string][string]
			strKey := fmt.Sprintf("%v", key)
			strValue := fmt.Sprintf("%v", value)
			hubMap1[strKey] = strValue
		}

		t.Run("while scanning", func (t *testing.T) {
			mockError := errors.New("mock - problem scanning")

			var u uint64 = 0
			r.On("Scan", mock.Anything, mock.Anything, mock.Anything).Return(redis.NewScanCmdResult(
				[]string {},
				u,
				mockError,
			)).Once()

			hubs, err := reg.GetHubs()
			assert.Empty(t, hubs)
			assert.Equal(t, mockError, err)
		})

		t.Run("while getting hubs, will skip the error", func (t *testing.T) {
			mockError := errors.New("mock - problem getting hubs")

			var u uint64 = 0
			r.On("Scan", mock.Anything, mock.Anything, mock.Anything).Return(redis.NewScanCmdResult(
				[]string { mockHub1RedisKey, },
				u,
				nil,
			)).Once()

			// Create a Mock Redis Pipeline and mock the Get calls to return back the proper hub
			mockPipe := new(mocks.Pipeliner)
			mockPipe.On("HGetAll", mockHub1RedisKey).Return(redis.NewStringStringMapResult(
				nil,
				mockError,
			)).Once()

			// Insert the mock Pipeline when TxPipeline is called
			r.On("TxPipeline").Return(mockPipe).Once()

			// Return back no error when it execs
			mockPipe.On("Exec").Return(
				[]redis.Cmder {},
				nil,
			).Once()

			hubs, err := reg.GetHubs()
			assert.Empty(t, hubs)
			assert.NoError(t, err)
		})

		t.Run("while calling exec", func (t *testing.T) {
			mockError := errors.New("mock - problem executing")

			var u uint64 = 0
			r.On("Scan", mock.Anything, mock.Anything, mock.Anything).Return(redis.NewScanCmdResult(
				[]string { mockHub1RedisKey, },
				u,
				nil,
			)).Once()

			// Create a Mock Redis Pipeline and mock the Get calls to return back the proper hub
			mockPipe := new(mocks.Pipeliner)
			mockPipe.On("HGetAll", mockHub1RedisKey).Return(redis.NewStringStringMapResult(
				hubMap1,
				nil,
			)).Once()

			// Insert the mock Pipeline when TxPipeline is called
			r.On("TxPipeline").Return(mockPipe).Once()

			// Return back no error when it execs
			mockPipe.On("Exec").Return(
				[]redis.Cmder {},
				mockError,
			).Once()

			hubs, err := reg.GetHubs()
			assert.Empty(t, hubs)
			assert.Equal(t, mockError, err)
		})
	})

	t.Run("when there is an error initializing the map it should silently continue", func (t *testing.T) {
		// Create a Mock Hub - #1
		mockHub1 := hub_registrytest.CreateMockTestHub()
		mockHub1RedisKey := hub_registry.HubDBIdentifier(mockHub1.ID)
		hubMap1 := make(map[string]string) // Purposely leaving empty

		// Create a Mock Hub - #2
		mockHub2 := hub_registrytest.CreateMockTestHub()
		mockHub2RedisKey := hub_registry.HubDBIdentifier(mockHub2.ID)
		hubMap2 := make(map[string]string)
		for key, value := range mockHub2.ToRedisMap() { // convert map[string]interface to map[string][string]
			strKey := fmt.Sprintf("%v", key)
			strValue := fmt.Sprintf("%v", value)
			hubMap2[strKey] = strValue
		}

		// Set up a test redis instance
		r := hub_registrytest.NewTestRedis()

		// Return back the hubs when scanning redis
		var u uint64 = 0
		r.On("Scan", mock.Anything, mock.Anything, mock.Anything).Return(redis.NewScanCmdResult(
			[]string {
				mockHub1RedisKey,
				mockHub2RedisKey,
			},
			u,
			nil,
		))

		// Create a Mock Redis Pipeline and mock the Get calls to return back the proper hub
		mockPipe := new(mocks.Pipeliner)
		mockPipe.On("HGetAll", mockHub1RedisKey).Return(redis.NewStringStringMapResult(
			hubMap1,
			nil,
		))
		mockPipe.On("HGetAll", mockHub2RedisKey).Return(redis.NewStringStringMapResult(
			hubMap2,
			nil,
		))

		// Return back no error when it execs
		mockPipe.On("Exec").Return(
			[]redis.Cmder {},
			nil,
		)

		// Insert the mock Pipeline when TxPipeline is called
		r.On("TxPipeline").Return(mockPipe)

		// Set up the registry
		reg := hub_registry.RedisRegistry{
			DBClient:  r,
			AppConfig: config.NewMock(),
		}

		// Make the test calls
		hubs, err := reg.GetHubs()
		assert.NoError(t, err)
		assert.Len(t, hubs, 1)
		hub_registrytest.AssertNotContainsHub(t, hubs, mockHub1, "should have skipped over the invalid hub")
		hub_registrytest.AssertContainsHub(t, hubs, mockHub2)
	})
}

func TestRegistry_GetHubById(t *testing.T) {
	t.Run("when a valid id", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()
		r := hub_registrytest.NewTestRedis()

		reg := hub_registry.RedisRegistry{
			DBClient:  r,
			AppConfig: config.NewMock(),
		}

		// Load a hub into redis
		err := reg.SaveHub(mockHub)
		require.NoError(t, err)

		res, err := reg.GetHubById(mockHub.ID)
		assert.NoError(t, err)
		hub_registrytest.AssertEqualHub(t, mockHub, res)
	})

	t.Run("when an invalid id", func (t *testing.T) {
		mockID := "thiswillnotexist"
		r := hub_registrytest.NewTestRedis()

		reg := hub_registry.RedisRegistry{
			DBClient: r,
		}

		res, err := reg.GetHubById(mockID)
		assert.Nil(t, res)
		assert.NoError(t, err)
	})

	t.Run("when an error from redis", func (t *testing.T) {
		mockID := "3856"
		mockErr := "problem connecting"
		r := hub_registrytest.NewTestRedis()

		r.On("HGetAll", hub_registry.HubDBIdentifier(mockID)).Return(redis.NewStringStringMapResult(
			nil,
			errors.New(mockErr),
		))

		reg := hub_registry.RedisRegistry{
			DBClient: r,
		}

		res, err := reg.GetHubById(mockID)
		assert.EqualError(t, err, mockErr)
		assert.Nil(t, res)
	})
}

func TestRegistry_DeleteHub(t *testing.T) {
	r := hub_registrytest.NewTestRedis()
	reg := hub_registry.RedisRegistry{
		DBClient:  r,
		AppConfig: config.NewMock(),
	}

	t.Run("successful if hub exists", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()
		err := reg.SaveHub(mockHub)
		assert.NoError(t, err)

		keys, err := r.Keys("*").Result()
		assert.NoError(t, err)
		assert.Len(t, keys, 1, "hub should exist within redis")

		err = reg.DeleteHub(mockHub.ID)
		assert.NoError(t, err)

		keys, err = r.Keys("*").Result()
		assert.NoError(t, err)
		assert.Len(t, keys, 0, "hub should have been deleted from redis")
	})

	t.Run("successful even if hub doesn't exist", func (t *testing.T) {
		// This is intentional because Redis' delete function: "A key is ignored if it does not exist."
		err := reg.DeleteHub("noExistID")
		assert.NoError(t, err)
	})

	t.Run("returns error if there is a redis error", func (t *testing.T) {
		mockErr := "there was an error deleting"
		r.On("Del", mock.Anything).Return(
			redis.NewIntResult(
				int64(1),
				errors.New(mockErr),
			),
		).Once()

		err := reg.DeleteHub("doesntmatter")
		assert.EqualError(t, err, mockErr)
	})
}

func TestRegistry_HubExists(t *testing.T) {
	r := hub_registrytest.NewTestRedis()
	reg := hub_registry.RedisRegistry{
		DBClient:  r,
		AppConfig: config.NewMock(),
	}

	t.Run("returns true if hub exists", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()

		// Load a mock hub into redis
		err := reg.SaveHub(mockHub)
		require.NoError(t, err)

		res, err := reg.HubExists(mockHub.ID)
		assert.NoError(t, err)
		assert.True(t, res)
	})

	t.Run("returns false if hub doesn't exist", func (t *testing.T) {
		res, err := reg.HubExists("iNoExist")
		assert.NoError(t, err)
		assert.False(t, res)
	})

	t.Run("returns error if redis returns an error", func (t *testing.T) {
		mockErr := "there was a problem"
		r.On("Exists", mock.Anything).Return(
			redis.NewIntResult(
				int64(0),
				errors.New(mockErr),
			),
		).Once()

		res, err := reg.HubExists("foobar")
		assert.EqualError(t, err, mockErr)
		assert.False(t, res)
	})
}

func TestRegistry_PauseHub(t *testing.T) {
	r := hub_registrytest.NewTestRedis()
	reg := hub_registry.RedisRegistry{
		DBClient:  r,
		AppConfig: config.NewMock(),
	}

	t.Run("pauses a hub", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()
		mockHub.Paused = false

		assert.False(t, mockHub.Paused)

		res, err := reg.PauseHub(mockHub)
		assert.NoError(t, err)
		assert.True(t, res.Paused)
	})

	t.Run("returns an error if hub is nil", func (t *testing.T) {
		res, err := reg.PauseHub(nil)
		assert.EqualError(t, err, "received a nil hub")
		assert.Nil(t, res)
	})
}

func TestRegistry_UnpauseHub(t *testing.T) {
	r := hub_registrytest.NewTestRedis()
	reg := hub_registry.RedisRegistry{
		DBClient:  r,
		AppConfig: config.NewMock(),
	}

	t.Run("unpauses a hub", func (t *testing.T) {
		mockHub := hub_registrytest.CreateMockTestHub()
		mockHub.Paused = true

		assert.True(t, mockHub.Paused)

		res, err := reg.UnpauseHub(mockHub)
		assert.NoError(t, err)
		assert.False(t, res.Paused)
	})

	t.Run("returns an error if hub is nil", func (t *testing.T) {
		res, err := reg.UnpauseHub(nil)
		assert.EqualError(t, err, "received a nil hub")
		assert.Nil(t, res)
	})
}

func TestRegistry_FindAvailableHub(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	reg := hub_registry.RedisRegistry{
		DBClient:  redisClient,
		AppConfig: config.NewMock(),
	}

	reg.DBClient.FlushAll() // clear the hubs
	// When no hub is available, returns nil
	res, err := reg.FindAvailableHub()
	assert.EqualError(t, err, "no available hub")

	// When a hub is in the registry but has no free slots, returns nil
	mockHub := hub_registrytest.CreateMockTestHub()
	mockHub.SlotCounts.Free = 0
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)

	res, err = reg.FindAvailableHub()
	assert.EqualError(t, err, "no available hub")

	// When a hub is available, returns it
	mockHub.SlotCounts.Free = 1
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)
	res, err = reg.FindAvailableHub()
	assert.NoError(t, err)
	hub_registrytest.AssertEqualHub(t, mockHub, res)

	/*
	When a hub is in the registry, but paused, returns nil
	 */
	assert.True(t, mockHub.SlotCounts.Free > 0) // Make sure it has free slots
	mockHub.Paused = true // Set to paused
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)
	res, err = reg.FindAvailableHub()
	assert.EqualError(t, err, "no available hub")

	/*
	When a hub is in the registry, but unpaused, returns it
	 */
	assert.True(t, mockHub.SlotCounts.Free > 0) // Make sure it has free slots
	mockHub.Paused = false // Set to unpaused
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)
	res, err = reg.FindAvailableHub()
	assert.NoError(t, err)
	hub_registrytest.AssertEqualHub(t, mockHub, res)

	/*
	  When more than one hub are in the registry but all are paused
	*/
	mockHub.Paused = true
	mockHub2 := hub_registrytest.CreateMockTestHub()
	mockHub2.SlotCounts.Free = 5
	mockHub2.Paused = true
	assert.True(t, mockHub.SlotCounts.Free > 0 && mockHub2.SlotCounts.Free > 0) // Make sure it has free slots
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)
	err = reg.SaveHub(mockHub2)
	assert.NoError(t, err)

	res, err = reg.FindAvailableHub()
	assert.EqualError(t, err, "no available hub")

	/*
	  When 1 hub gets unpaused, gets returned
	 */
	mockHub.Paused = false // Unpause the first hub, leaving the second paused
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)

	res, err = reg.FindAvailableHub()
	assert.NoError(t, err)
	hub_registrytest.AssertEqualHub(t, mockHub, res)

	// Try unpausing the second hub, to make sure it's not just an ordering reason
	mockHub.Paused = true
	mockHub2.Paused = false
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)
	err = reg.SaveHub(mockHub2)
	assert.NoError(t, err)

	res, err = reg.FindAvailableHub()
	assert.NoError(t, err)
	hub_registrytest.AssertEqualHub(t, mockHub2, res)

	t.Run("returns an error if GetHubs() returns an error", func (t *testing.T) {
		mockErr := "redis error scanning"

		r := hub_registrytest.NewTestRedis()
		reg := hub_registry.RedisRegistry{
			DBClient: r,
		}

		r.On("Scan", mock.Anything, mock.Anything, mock.Anything).Return(redis.NewScanCmdResult(
			[]string{},
			uint64(0),
			errors.New(mockErr),
		)).Once()

		res, err := reg.FindAvailableHub()
		assert.EqualError(t, err, mockErr)
		assert.Nil(t, res)
	})

	t.Run("distributes the tests", func (t *testing.T) {
		t.Run("when only one hub", func (t *testing.T) {
			reg.DBClient.FlushAll()
			reg.LastHubIndex = -1 // reset the index to "never ran"

			mockHub = hub_registrytest.CreateMockTestHub()
			mockHub.SlotCounts.Free = 5
			err = reg.SaveHub(mockHub)
			require.NoError(t, err)

			allHubs, err := reg.GetHubs()
			require.NoError(t, err)
			require.True(t, len(allHubs) == 1, "PreCondition: Hubs should have one hub in it")

			t.Run("returns the same hub when available on both calls", func (t *testing.T) {
				numberOfCalls := 2
				for i := 0; i < numberOfCalls; i++ {
					t.Run(fmt.Sprintf("%d", i), func (t *testing.T) { // print the index #
						res, err := reg.FindAvailableHub()
						assert.NoError(t, err)
						hub_registrytest.AssertEqualHub(t, res, allHubs[0])
					})
				}
			})
		})

		t.Run("when multiple hubs", func (t *testing.T) {
			reg.DBClient.FlushAll()
			reg.LastHubIndex = -1 // reset the index to "never ran"

			mockHub = hub_registrytest.CreateMockTestHub()
			mockHub.SlotCounts.Free = 5
			err = reg.SaveHub(mockHub)
			require.NoError(t, err)

			mockHub2 = hub_registrytest.CreateMockTestHub()
			mockHub2.SlotCounts.Free = 5
			err = reg.SaveHub(mockHub2)
			require.NoError(t, err)

			allHubs, err := reg.GetHubs()
			require.NoError(t, err)
			require.True(t, len(allHubs) == 2, "PreCondition: Hubs should have two hubs in it")

			t.Run("First request returns Hub 1", func (t *testing.T) {
				res, err := reg.FindAvailableHub()
				assert.NoError(t, err)
				hub_registrytest.AssertEqualHub(t, res, allHubs[0])
			})

			t.Run("Second request returns Hub 2", func (t *testing.T) {
				res, err := reg.FindAvailableHub()
				assert.NoError(t, err)
				hub_registrytest.AssertEqualHub(t, res, allHubs[1])
			})

			t.Run("Third request returns Hub 1", func (t *testing.T) {
				res, err := reg.FindAvailableHub()
				assert.NoError(t, err)
				hub_registrytest.AssertEqualHub(t, res, allHubs[0])
			})

			t.Run("returns to beginning if last hub is full", func (t *testing.T) {
				reg.LastHubIndex = 0
				hub := allHubs[1]
				hub.SlotCounts.Free = 0
				err = reg.SaveHub(hub)
				require.NoError(t, err)

				res, err = reg.FindAvailableHub()
				assert.NoError(t, err)
				hub_registrytest.AssertEqualHub(t, allHubs[0], res)
			})
		})
	})
}

func TestRegistry_PollForAvailableHub(t *testing.T) {
	goRoutineSleepDuration := time.Duration(time.Millisecond / 2) // Go Routines that run async might take a bit to finish. Determines how long to sleep

	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	// Create a Mock Metric Writer and Instrumentor so that we can make sure we're sending duration metrics
	mockMetricWriter := &instrumentorMocks.MetricWriter{} // The mock Metric Writer
	mockMetricWriter.On("PutMetricData", mock.Anything).Return(&cloudwatch.PutMetricDataOutput{}, nil)
	mockInstrumentor := &instrumentor.Instrumentor{
		Logger: config.NewMock().Logger,
		MetricClient: mockMetricWriter,
		AutoScalingGroupName: "localtest",
		DryRun: false,
	}
	mockInstrumentor.MetricClient = mockMetricWriter

	mockClock := clockwork.NewRealClock()
	reg := hub_registry.RedisRegistry{
		DBClient:     redisClient,
		AppConfig:    config.NewMock(),
		Instrumentor: mockInstrumentor,
	}
	reg.AppConfig.Clock = mockClock

	reg.DBClient.FlushAll() // clear the hubs

	/*
	Returns an error when no hubs are available and the timeout was reached
	 */

	_, err = reg.PollForAvailableHub(time.Millisecond * 100, time.Millisecond * 300)
	assert.EqualError(t, err, "no available hub")
	if !testing.Short() {
		time.Sleep(goRoutineSleepDuration)
		mockMetricWriter.AssertNumberOfCalls(t, "PutMetricData", 1) // should have logged duration to cloudwatch
		mockMetricWriter.Calls = nil // Reset the number of calls
	}

	/*
	Returns a hub when immediately available
	 */

	 // Create an available hub, add it to the Hubs
	mockHub := hub_registrytest.CreateMockTestHub()
	mockHub.SlotCounts.Free = 1
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)

	res, err := reg.PollForAvailableHub(time.Millisecond * 100, time.Millisecond * 100)
	assert.NoError(t, err)
	hub_registrytest.AssertEqualHub(t, mockHub, res)
	if !testing.Short() {
		time.Sleep(goRoutineSleepDuration) // runs async in a go routine... sleep a bit so it can have time to run
		mockMetricWriter.AssertNumberOfCalls(t, "PutMetricData", 1) // should have logged duration to cloudwatch
		mockMetricWriter.Calls = nil // Reset the number of calls
	}

	/*
	Returns a hub mid poll when one becomes available
	 */
	// Set the slot count to 0
	mockHub.SlotCounts.Free = 0
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)

	// Set up a go routine to poll for hubs and run async
	hubChannel := make(chan *hub_registry.Hub)
	errorChannel := make(chan error)
	go func (c chan error, hubC chan *hub_registry.Hub) {
		res, err := reg.PollForAvailableHub(time.Millisecond * 100, time.Millisecond * 500)
		c <- err
		hubC <- res

	}(errorChannel, hubChannel)

	// Sleep for a few seconds before updating the hub
	time.Sleep(time.Millisecond * 300)

	// Update the hub to available
	mockHub.SlotCounts.Free = 1
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)

	// Wait to hear back from goroutine channel
	err = <-errorChannel
	hubResult := <- hubChannel

	assert.NoError(t, err)
	hub_registrytest.AssertEqualHub(t, mockHub, hubResult)
	if !testing.Short() {
		time.Sleep(goRoutineSleepDuration) // runs async in a go routine... sleep a bit so it can have time to run
		mockMetricWriter.AssertNumberOfCalls(t, "PutMetricData", 1) // should have logged duration to cloudwatch
		mockMetricWriter.Calls = nil // Reset the number of calls
	}
}

func TestRegistry_SaveHubInRedis(t *testing.T) {
	s, err := miniredis.Run()
	if err != nil {
		panic(err)
	}
	defer s.Close()

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})

	mockClock := clockwork.NewRealClock()
	reg := hub_registry.RedisRegistry{
		DBClient:           redisClient,
		AppConfig:          config.NewMock(),
		HubConsideredStale: time.Hour,
	}
	reg.AppConfig.Clock = mockClock

	reg.DBClient.FlushAll() // clear the hubs

	/*
	should get saved into redis
	 */
	mockHub := hub_registrytest.CreateMockTestHub()

	res, err := reg.DBClient.Keys("*").Result()
	assert.NoError(t, err)
	assert.Equal(t, 0, len(res), "should be empty")

	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)
	res, err = reg.DBClient.Keys("*").Result()
	assert.NoError(t, err)
	assert.Equal(t, 1, len(res), "should have 1 key")

	hubFromDB, err := reg.DBClient.HGetAll(hub_registry.HubDBIdentifier(mockHub.ID)).Result()
	assert.NoError(t, err)
	assert.NotEmpty(t, hubFromDB)
	assert.Equal(t, mockHub.ID, hubFromDB["id"])
	assert.Equal(t, strconv.Itoa(mockHub.SlotCounts.Free), hubFromDB["slotcounts:free"])

	/*
	should update
	 */
	mockHub.SlotCounts.Free += 1
	err = reg.SaveHub(mockHub)
	assert.NoError(t, err)

	res, err = reg.DBClient.Keys("*").Result()
	assert.NoError(t, err)
	assert.Equal(t, 1, len(res), "should still only have 1 key")

	hubFromDB, err = reg.DBClient.HGetAll(hub_registry.HubDBIdentifier(mockHub.ID)).Result()
	assert.NoError(t, err)
	assert.NotEmpty(t, hubFromDB)
	assert.Equal(t, mockHub.ID, hubFromDB["id"])
	assert.Equal(t, strconv.Itoa(mockHub.SlotCounts.Free), hubFromDB["slotcounts:free"])

	/*
	should expire
	 */
	hubTTL, err := reg.DBClient.TTL(hub_registry.HubDBIdentifier(mockHub.ID)).Result()
	assert.Equal(t, reg.HubConsideredStale, hubTTL)
	s.FastForward(reg.HubConsideredStale)

	res, err = reg.DBClient.Keys("*").Result()
	assert.NoError(t, err)
	assert.Equal(t, 0, len(res), "should be empty")

	hubFromDB, err = reg.DBClient.HGetAll(hub_registry.HubDBIdentifier(mockHub.ID)).Result()
	assert.NoError(t, err)
	assert.Empty(t, hubFromDB, "the hub should no longer exist")

	t.Run("returns error if no dbclient", func (t *testing.T) {
		reg := hub_registry.RedisRegistry{}
		err := reg.SaveHub(hub_registrytest.CreateMockTestHub())
		assert.EqualError(t, err, "no Redis Client associated with registry")
	})

	t.Run("returns an error if hub is nil", func (t *testing.T) {
		reg := hub_registry.RedisRegistry{
			DBClient: hub_registrytest.NewTestRedis(),
		}

		err := reg.SaveHub(nil)
		assert.EqualError(t, err, "nil hub passed to method")
	})

	t.Run("returns an error if redis returns error", func (t *testing.T) {
		r := hub_registrytest.NewTestRedis()
		reg := hub_registry.RedisRegistry{
			DBClient:  r,
			AppConfig: config.NewMock(),
		}

		t.Run("HMSet", func (t *testing.T) {
			mockErr := "error on hmset"
			mockPipe := new(mocks.Pipeliner)
			mockPipe.On("HMSet", mock.Anything, mock.Anything).Return(redis.NewStatusResult(
				"success",
				errors.New(mockErr),
			)).Once()

			// Return back no error when it execs
			mockPipe.On("Exec").Return(
				[]redis.Cmder {},
				nil,
			).Once()

			// Insert the mock Pipeline when TxPipeline is called
			r.On("TxPipeline").Return(mockPipe).Once()

			err := reg.SaveHub(hub_registrytest.CreateMockTestHub())
			assert.EqualError(t, err, mockErr)
		})

		t.Run("Exec", func (t *testing.T) {
			mockErr := "error on exec"
			mockPipe := new(mocks.Pipeliner)
			mockPipe.On("HMSet", mock.Anything, mock.Anything).Return(redis.NewStatusResult(
				"success",
				nil,
			)).Once()

			// Return back no error when it execs
			mockPipe.On("Exec").Return(
				[]redis.Cmder {},
				errors.New(mockErr),
			).Once()

			// Insert the mock Pipeline when TxPipeline is called
			r.On("TxPipeline").Return(mockPipe).Once()

			err := reg.SaveHub(hub_registrytest.CreateMockTestHub())
			assert.EqualError(t, err, mockErr)
		})
	})
}
