package hub_registrytest

import (
	"bytes"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"encoding/json"
	"fmt"
	"github.com/alicebob/miniredis/v2"
	"github.com/bxcodec/faker"
	"github.com/davecgh/go-spew/spew"
	"github.com/elliotchance/redismock/v6"
	"github.com/go-redis/redis"
	"github.com/gorilla/mux"
	"github.com/stretchr/testify/assert"
	"io"
	"io/ioutil"
	"log"
	"math/rand"
	"net/http"
	"net/http/httptest"
	"reflect"
	"strconv"
	"testing"
	"text/template"
	"time"
)

// Populates test data into the Hub array
// Provide a count for how many hubs to create and the registry, so it can save the hub
// Returns an array of the Mock Hubs that were created
func LoadTestData(count int, reg *hub_registry.RedisRegistry) []*hub_registry.Hub {
	var hubs []*hub_registry.Hub
	for i := 0; i < count; i++ {
		mockHub := CreateMockTestHub()
		redisKey := hub_registry.HubDBIdentifier(mockHub.ID)
		reg.DBClient.HMSet(redisKey, mockHub.ToRedisMap())
		hubs = append(hubs, mockHub)
	}
	return hubs
}

// Creates a Mocked Hub
func CreateMockTestHub() *hub_registry.Hub {
	var testHub hub_registry.Hub // Initialize the hub

	// Add a provider called Custom Port, which will assign a value through 9999
	_ = faker.AddProvider("customPort", func(v reflect.Value) (interface{}, error) {
		port := rand.Intn(9999)
		strPort := strconv.Itoa(port)
		return strPort, nil
	})

	// Create the Hub with fake data
	err := faker.FakeData(&testHub)
	if err != nil {
		log.Fatalf("error creating fake data %v", err)
	}

	// Set the Created/Updated at values
	testHub.CreatedAt = time.Now()
	testHub.UpdatedAt = time.Now()
	testHub.Healthy = true

	return &testHub
}

// Creates a request to the hub api
// If provided an ID, it will append that to the route
// Provide a Method to request as
// Provide a body to send as the request. Nil if no data to send
func CreateTestRequest(id string, method string, body io.Reader) *http.Request {
	var url string
	baseUrl := "/cbg/api/hub/registry"

	// If there is an ID, append it to the URL
	if id == "" {
		url = baseUrl
	} else {
		url = baseUrl + fmt.Sprintf("/%s", id)
	}

	// Create the Request Object
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		log.Fatalf("Encountered error while creating request: %v", err)
	}

	return req
}

// Creates a request to the hub api, with the ID Parameter set (helps with Gorilla Mux testing)
// If provided an ID, it will append that to the route
// Provide a Method to request as
// Provide a body to send as the request. Nil if no data to send
func CreateTestRequestWithParams(id string, method string, body io.Reader) *http.Request {
	req := CreateTestRequest(id, method, body)
	// If there is an ID, set mock conditions so that Gorilla Mux can read the ID
	if id != "" {
		req = mux.SetURLVars(req, map[string]string{"id": id})
	}

	return req
}

// Helper method that will mock an HTTP Request and send a provided hub to the CreateHub handler
// This is used as a shortcut to handle all of the marshalling of data
func PostRequestToCreateHub(hub *hub_registry.Hub, api hub_registry.Api) *http.Response {
	w := httptest.NewRecorder()
	bodyData, err := json.Marshal(&hub)
	if err != nil {
		log.Fatalf("problem marshaling data: %v", err)
	}
	body := bytes.NewReader(bodyData)
	req := CreateTestRequestWithParams(hub.ID, http.MethodPost, body)

	api.CreateHub(w, req)
	return w.Result()
}

// Helper function for the assert functions that will get a hub by its ID
func GetHubForAssert(hubs []*hub_registry.Hub, expectedHub *hub_registry.Hub) *hub_registry.Hub {
	for _, hub := range hubs {
		if hub.ID == expectedHub.ID {
			return hub
		}
	}
	return nil
}

// Helper function for assert.NotContains. Due to timestamps, cannot use the base assert.NotContains
func AssertNotContainsHub(t *testing.T, hubs []*hub_registry.Hub, expectedHub *hub_registry.Hub, msgAndArgs ...interface{}) {
	foundHub := GetHubForAssert(hubs, expectedHub)
	if foundHub != nil {
		t.Errorf("Expected not to find Hub in Hubs.\nExpected: %v\n\nActual: %v\nMessages: %v", spew.Sdump(expectedHub), spew.Sdump(hubs), msgAndArgs)
	}
}

// Helper function for assert.Contains. Due to timestamps, cannot use the base assert.NotContains
func AssertContainsHub(t *testing.T, hubs []*hub_registry.Hub, expectedHub *hub_registry.Hub, msgAndArgs ...interface{}) {
	foundHub := GetHubForAssert(hubs, expectedHub)

	if foundHub == nil {
		t.Errorf("The hubs did not contain the expected hub\nExpected: %v\n\nWithin: %v\nMessages: %v", spew.Sdump(expectedHub), spew.Sdump(hubs), msgAndArgs)
	}

	// Assert every field
	AssertEqualHub(t, expectedHub, foundHub)
}

// Asserts every field of the struct, without looking at Created/Updated times
func AssertEqualHub(t *testing.T, expectedHub *hub_registry.Hub, actualHub *hub_registry.Hub) {
	// Assert every field is equal
	assert.Equal(t, expectedHub.ID, actualHub.ID)
	assert.Equal(t, expectedHub.IP, actualHub.IP)
	assert.Equal(t, expectedHub.Port, actualHub.Port)
	assert.Equal(t, expectedHub.ClusterName, actualHub.ClusterName)
	assert.Equal(t, expectedHub.SlotCounts, actualHub.SlotCounts)
}

// Helper function that gets hub and handles error checking
func GetHubs(t *testing.T, reg *hub_registry.RedisRegistry) []*hub_registry.Hub {
	hubs, err := reg.GetHubs()
	assert.NoError(t, err)
	return hubs
}

func NewTestRedis() *redismock.ClientMock {
	mr, err := miniredis.Run()
	if err != nil {
		panic(err)
	}

	client := redis.NewClient(&redis.Options{
		Addr: mr.Addr(),
	})

	return redismock.NewNiceMock(client)
}

// Test Helper to create a json object for the hub
// hub The hub to transform to json
// createdAt The time to specify for the hub created at. Supply blank "" to not include
// updatedAt The time to specify for the hub updated at. Supply blank "" to not include
// paused A string of a boolean to specify if the hub is paused. Supply blank "" to not include
// healthy A string of a boolean to specify if the hub is healthy. Supply blank "" to not include
func GetHubJson(hub *hub_registry.Hub, createdAt string, updatedAt string, paused string, healthy string) string {
	const hubTemplate = `
{
 "id": "{{.ID}}",
 "IP": "{{.IP}}",
 "Port": "{{.Port}}",
 "clusterName": "{{.ClusterName}}",
 "Hostname": "{{.Hostname}}",
 "SlotCounts": {
   "Free": {{.SlotCounts.Free}},
   "Total": {{.SlotCounts.Total}}
 }
`
	t := template.Must(template.New("hubTemplate").Parse(hubTemplate))
	var tpl bytes.Buffer
	err := t.Execute(&tpl, hub)
	if err != nil {
		panic(err)
	}

	templateStr := tpl.String()

	// Append any additional options if they were provided
	// This helps us test different inputs
	if len(paused) > 0 {
		templateStr += fmt.Sprintf(", \"paused\": %s", paused)
	}
	if len(healthy) > 0 {
		templateStr += fmt.Sprintf(", \"healthy\": %s", healthy)
	}
	if len(createdAt) > 0 {
		templateStr += fmt.Sprintf(", \"created_at\": %s", createdAt)
	}
	if len(updatedAt) > 0 {
		templateStr += fmt.Sprintf(", \"updated_at\": %s", updatedAt)
	}

	// Close out the json object before returning
	templateStr += " }"
	return templateStr
}

// Calls getHub but returns a Reader instead of a string
// hub The hub to transform to json
// createdAt The time to specify for the hub created at. Supply blank "" to not include
// updatedAt The time to specify for the hub updated at. Supply blank "" to not include
// paused A string of a boolean to specify if the hub is paused. Supply blank "" to not include
// healthy A string of a boolean to specify if the hub is healthy. Supply blank "" to not include
func GetHubJsonReader(hub *hub_registry.Hub, createdAt string, updatedAt string, paused string, healthy string) io.ReadCloser {
	jsonData := GetHubJson(hub, createdAt, updatedAt, paused, healthy)
	return ioutil.NopCloser(bytes.NewReader([]byte(jsonData)))
}
