package instrumentor

import (
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor/mocks"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"testing"
)

func TestNew(t *testing.T) {
	t.Run("sets the proper values", func (t *testing.T) {
		logger, _ := test.NewNullLogger()
		client    := &mocks.MetricWriter{}
		asgName   := "myTestASG"

		resp := New(logger, client, asgName)
		assert.Equal(t, logger, resp.Logger)
		assert.Equal(t, client, resp.MetricClient)
		assert.Equal(t, asgName, resp.AutoScalingGroupName)
		assert.Equal(t, false, resp.DryRun)
	})
}

func TestInstrumentor_PutMetricData(t *testing.T) {
	logger, loggerHook := test.NewNullLogger()
	logger.Level = logrus.DebugLevel // set to the debug level

	t.Run("when dry run, does not write to MetricWriter", func (t *testing.T) {
		client := &mocks.MetricWriter{}
		instrumentor := Instrumentor{
			MetricClient:         client,
			AutoScalingGroupName: "",
			Logger:               logger,
			DryRun:               true,
		}

		loggerHook.Reset()
		_, err := instrumentor.PutMetricData(nil)
		assert.NoError(t, err)
		client.AssertNotCalled(t, "PutMetricData")
		assert.Equal(t, 1, len(loggerHook.Entries)) // wrote to the logger
	})

	t.Run("when not dry run, writes to metric writer", func (t *testing.T) {
		client := &mocks.MetricWriter{}
		client.On("PutMetricData", mock.Anything).Return(&cloudwatch.PutMetricDataOutput{}, nil).Once()
		instrumentor := Instrumentor{
			MetricClient:         client,
			AutoScalingGroupName: "",
			Logger:               logger,
			DryRun:               false,
		}

		loggerHook.Reset()
		mockInput := &cloudwatch.PutMetricDataInput{}
		_, err := instrumentor.PutMetricData(mockInput)
		assert.NoError(t, err)
		client.AssertCalled(t, "PutMetricData", mockInput)
		assert.Equal(t, 0, len(loggerHook.Entries)) // did not write to logger
	})
}
