package sdk

import (
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
)

const RegistryPath = "/cbg/api/hub/registry"

// This will return all hubs attached to the registry
func (c *Client) GetHubs() ([]*hub_registry.Hub, error) {
	var hubs []*hub_registry.Hub

	url := fmt.Sprintf("%s%s", c.RootURL(), RegistryPath)
	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil { return hubs, err }

	resp, err := processRequestWithLog(c.HttpClient, req)
	if err != nil { return hubs, err }

	defer resp.Body.Close()
	respBody, err := ioutil.ReadAll(resp.Body)
	if err != nil { return hubs, err }

	err = json.Unmarshal(respBody, &hubs)
	if err != nil { return hubs, err }

	return hubs, nil
}

// Returns a single Healthy Hub object from the registry. Unhealthy objects will be filtered out.
// Provide the clusterName you are looking for
func (c *Client) GetHubByClusterName(clusterName string) (*hub_registry.Hub, error) {
	hubs, err := c.GetHubs()
	if err != nil { return nil, err }

	// Loop through all hubs and search for the cluster name
	// Unfortunately there's not yet a server api to look up by cluster name
	for _, hub := range hubs {
		if hub.Healthy && hub.ClusterName == clusterName {
			return hub, nil
		}
	}

	return nil, fmt.Errorf("unable to find healthy hub by name: %s", clusterName)
}

// Pauses a Hub in the registry by a provided clusterName
func (c *Client) PauseHubByClusterName(clusterName string) error {
	return c.modifyHubPauseByClusterName(clusterName, true)
}

// Unpauses a Hub in the registry by a provided clusterName
func (c *Client) UnpauseHubByClusterName(clusterName string) error {
	return c.modifyHubPauseByClusterName(clusterName, false)
}

// Function that modifies a hub pause/unpause
// id for the Hub's ID (not cluster name)
// pause True if Pause, False if Unpause
func (c *Client) modifyHubPause(id string, pause bool) error {
	actionName := ""
	if pause {
		actionName = "pause"
	} else {
		actionName = "unpause"
	}

	url := fmt.Sprintf("%s%s/%s/%s", c.RootURL(), RegistryPath, id, actionName)
	req, err := http.NewRequest(http.MethodPut, url, nil)
	if err != nil { return err }

	_, err = processRequestWithLog(c.HttpClient, req)
	if err != nil { return err }
	return nil
}

// Function that modifies a hub pause/unpause
// clusterName for the Hub's clusterName (not ID)
// pause True if Pause, False if Unpause
func (c *Client) modifyHubPauseByClusterName(clusterName string, pause bool) error {
	hub, err := c.GetHubByClusterName(clusterName)
	if err != nil {
		return err
	}

	return c.modifyHubPause(hub.ID, pause)
}

// Processes a given request with logging and checks if the response has a healthy status code
// httpClient The HTTP Client to use for the request
// req The HTTP Request to make
// Returns an error if there was an issue sending the request, or the response was a non-successful code (2xx)
func processRequestWithLog(httpClient *http.Client, req *http.Request) (*http.Response, error)  {
	log.Printf("-> %s\n", req.URL)
	resp, err := httpClient.Do(req)
	if err != nil { return resp, err }
	log.Printf("<- [%d] %s\n", resp.StatusCode, req.URL)

	if resp.StatusCode < 200 || resp.StatusCode > 299 {
		return resp, fmt.Errorf("unexpected status code: %d", resp.StatusCode)
	}

	return resp, nil
}
