package sdk

import (
	hub_registrytest "code.justin.tv/qe/grid_router/src/pkg/hub_registry/test"
	"encoding/json"
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"log"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"
	"time"
)

func TestClient_GetHubs(t *testing.T) {
	t.Run("returns a single hub", func (t *testing.T) {
		expectedHub := hub_registrytest.CreateMockTestHub()

		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			hubJson, err := json.Marshal(&expectedHub)
			require.NoError(t, err)

			_, err = fmt.Fprintf(w, "[\n%s\n]", hubJson)
			require.NoError(t, err)
		}))
		defer server.Close()

		client := createTestClient(server.URL)
		resp, err := client.GetHubs()
		assert.NoError(t, err)

		assert.Len(t, resp, 1)
		hub_registrytest.AssertContainsHub(t, resp, expectedHub)
	})

	t.Run("returns multiple hubs", func (t *testing.T) {
		expectedHub1 := hub_registrytest.CreateMockTestHub()
		expectedHub2 := hub_registrytest.CreateMockTestHub()

		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			hubJson1, err := json.Marshal(&expectedHub1)
			require.NoError(t, err)

			hubJson2, err := json.Marshal(&expectedHub2)
			require.NoError(t, err)

			_, err = fmt.Fprintf(w, "[\n%s,\n%s\n]", hubJson1, hubJson2)
			require.NoError(t, err)
		}))
		defer server.Close()

		client := createTestClient(server.URL)
		resp, err := client.GetHubs()
		assert.NoError(t, err)
		assert.Len(t, resp, 2)
		hub_registrytest.AssertContainsHub(t, resp, expectedHub1)
		hub_registrytest.AssertContainsHub(t, resp, expectedHub2)
	})

	t.Run("returns error if the http server returns an error", func (t *testing.T) {
		httpError     := "test unexpected error"
		httpErrorCode := 500

		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			http.Error(w, httpError, httpErrorCode)
		}))
		defer server.Close()

		client := createTestClient(server.URL)
		resp, err := client.GetHubs()
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("unexpected status code: %d", httpErrorCode))
		assert.Empty(t, resp)
	})

	t.Run("returns an error if there is a problem unmarshalling data", func (t *testing.T) {
		server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			fmt.Fprintln(w, "not json data")
		}))
		defer server.Close()

		client := createTestClient(server.URL)
		resp, err := client.GetHubs()
		assert.Error(t, err)
		assert.Empty(t, resp)
	})
}

func TestClient_GetHubByClusterName(t *testing.T) {
	expectedHub := hub_registrytest.CreateMockTestHub()

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		hubJson, err := json.Marshal(&expectedHub)
		require.NoError(t, err)

		_, err = fmt.Fprintf(w, "[\n%s\n]", hubJson)
		require.NoError(t, err)
	}))
	defer server.Close()

	client := createTestClient(server.URL)

	t.Run("given a valid cluster name", func (t *testing.T) {
		validClusterName := expectedHub.ClusterName

		resp, err := client.GetHubByClusterName(validClusterName)
		assert.NoError(t, err)
		require.NotNil(t, resp)
		hub_registrytest.AssertEqualHub(t, expectedHub, resp)
	})

	t.Run("given an unhealthy cluster name, returns an error", func (t *testing.T) {
		validClusterName := expectedHub.ClusterName
		expectedHub.Healthy = false

		resp, err := client.GetHubByClusterName(validClusterName)
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("unable to find healthy hub by name: %s", validClusterName))
		assert.Nil(t, resp)
	})

	t.Run("given a invalid cluster name, returns an error", func (t *testing.T) {
		invalidCluserName := "invalid-does-not-exist"
		resp, err := client.GetHubByClusterName(invalidCluserName)
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("unable to find healthy hub by name: %s", invalidCluserName))
		assert.Nil(t, resp)
	})
}

func TestClient_PauseHubByClusterName(t *testing.T) {
	serverShouldError := false
	serverErrorCode   := 500
	receivedPausePath := ""
	expectedHub := hub_registrytest.CreateMockTestHub()

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if serverShouldError {
			http.Error(w, "testing error", serverErrorCode)
			return
		}

		// If a pause / unpause request
		if r.RequestURI == fmt.Sprintf("/cbg/api/hub/registry/%s/pause", expectedHub.ID) {
			receivedPausePath = r.RequestURI
			w.WriteHeader(200)
			return
		}

		// other requests
		hubJson, err := json.Marshal(&expectedHub)
		require.NoError(t, err)

		_, err = fmt.Fprintf(w, "[\n%s\n]", hubJson)
		require.NoError(t, err)
	}))
	defer server.Close()

	client := createTestClient(server.URL)

	t.Run("returns no error when successful", func (t *testing.T) {
		serverShouldError = false

		err := client.PauseHubByClusterName(expectedHub.ClusterName)
		assert.NoError(t, err)
		expectedPath := fmt.Sprintf("/cbg/api/hub/registry/%s/pause", expectedHub.ID)
		assert.Equal(t, expectedPath, receivedPausePath)
	})

	t.Run("returns error if cluster name doesn't exist", func (t *testing.T) {
		serverShouldError = false
		clusterName := "does-not-exist"

		err := client.PauseHubByClusterName(clusterName)
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("unable to find healthy hub by name: %s", clusterName))
	})

	t.Run("returns error if server returns error", func (t *testing.T) {
		serverShouldError = true
		err := client.PauseHubByClusterName(expectedHub.ClusterName)
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("unexpected status code: %d", serverErrorCode))
	})
}

func TestClient_UnpauseHubByClusterName(t *testing.T) {
	serverShouldError := false
	serverErrorCode   := 500
	receivedPausePath := ""
	expectedHub := hub_registrytest.CreateMockTestHub()

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if serverShouldError {
			http.Error(w, "testing error", serverErrorCode)
			return
		}

		// If a pause / unpause request
		if r.RequestURI == fmt.Sprintf("/cbg/api/hub/registry/%s/unpause", expectedHub.ID) {
			receivedPausePath = r.RequestURI
			w.WriteHeader(200)
			return
		}

		// other requests
		hubJson, err := json.Marshal(&expectedHub)
		require.NoError(t, err)

		_, err = fmt.Fprintf(w, "[\n%s\n]", hubJson)
		require.NoError(t, err)
	}))
	defer server.Close()

	client := createTestClient(server.URL)

	t.Run("returns no error when successful", func (t *testing.T) {
		serverShouldError = false

		err := client.UnpauseHubByClusterName(expectedHub.ClusterName)
		assert.NoError(t, err)
		expectedPath := fmt.Sprintf("/cbg/api/hub/registry/%s/unpause", expectedHub.ID)
		assert.Equal(t, expectedPath, receivedPausePath)
	})

	t.Run("returns error if cluster name doesn't exist", func (t *testing.T) {
		serverShouldError = false
		clusterName := "does-not-exist"

		err := client.UnpauseHubByClusterName(clusterName)
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("unable to find healthy hub by name: %s", clusterName))
	})

	t.Run("returns error if server returns error", func (t *testing.T) {
		serverShouldError = true
		err := client.UnpauseHubByClusterName(expectedHub.ClusterName)
		assert.Error(t, err)
		assert.EqualError(t, err, fmt.Sprintf("unexpected status code: %d", serverErrorCode))
	})
}

// Creates a client for tests that parses a url given by the test http server
func createTestClient(serverURL string) *Client {
	serverParsedUrl, err := url.Parse(serverURL)
	if err != nil {
		log.Fatalf("problem parsing %s - %v", serverURL, err)
	}

	return &Client{
		Scheme: serverParsedUrl.Scheme,
		Host:   serverParsedUrl.Host,
		HttpClient: &http.Client{
			Timeout: time.Second * 5,
		},
	}
}
