package session

import (
	"bytes"
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"io/ioutil"
	"net/http"
	"net/http/httputil"
	"strconv"
	"strings"
	"time"
)

type Handler struct {
	Registry  *hub_registry.RedisRegistry
	apiKeys   []string
	appConfig *config.Config
	proxy     *httputil.ReverseProxy
}

// Creates a new Session Handler
func NewHandler(registry *hub_registry.RedisRegistry, apiKeys []string,
	appConfig *config.Config) *Handler {
	return &Handler{
		Registry:  registry,
		apiKeys:   apiKeys,
		appConfig: appConfig,
		proxy:     NewProxy(appConfig, registry),
	}
}

// Handles requests for Selenium Sessions
func (sh *Handler) Handle(w http.ResponseWriter, req *http.Request) {
	if !sh.IsAuthorized(req, sh.apiKeys) {
		sh.appConfig.Logger.Warn("Returning as request was not authorized.")
		w.Header().Set("WWW-Authenticate", `Basic realm="Cross Browser Grid"`) // Tell browsers it needs to authenticate
		http.Error(w, "Not authorized", http.StatusUnauthorized)
		return
	} else {
		sh.appConfig.Logger.Debug("Proceeding as request was authorized")
	}

	// Set the Accept-Encoding header to identity.
	// TODO: Support multiple encodings requested by client. QE-2425
	req.Header.Set("Accept-Encoding", "identity")

	// Serve the HTTP Request from the Proxy
	// This will run the transport.RoundTrip function within proxy.go
	sh.proxy.ServeHTTP(w, req)
}

// Determines if a request is authorized via an API Key
func (sh *Handler) IsAuthorized(req *http.Request, apiKeys []string) bool {
	username, password, authOK := req.BasicAuth()

	// If the authentication wasn't provided, return false
	if !authOK {
		sh.appConfig.Logger.Info("Authentication Failed: basic auth method returned false for 'ok'")
		return false
	}

	// Look through all stored api keys
	for _, element := range apiKeys {
		if password == element {
			sh.appConfig.Logger.Infof("Authentication Succeeded: username %s", username)
			return true
		}
	}

	sh.appConfig.Logger.Warnf("Authentication Failed: username %s did not provide a valid auth key", username)
	return false // If it got here, it didn't find anything
}

// Used for when a New Session request comes in, the hook before it's passed to a Hub
// Responsible for finding an available hub and preparing the request for that hub
func handleNewSessionBefore(req *Request) error {
	// Find an available hub for the new session
	hub, err := req.Registry.PollForAvailableHub(time.Second, time.Second * 60)
	if err != nil || hub == nil {
		return errors.New("unable to find an available hub")
	}

	// Create a new session with the hub
	req.Session = &Session{
		Hub: hub,
		AppConfig: req.AppConfig,
	}

	// Decrement the registry free slot with the session for the hub
	req.AppConfig.Logger.Debugf("Removing 1 FreeSlot to hub: %+v", hub)
	hub.SlotCounts.Free -= 1
	err = req.Registry.SaveHub(hub)
	return err
}

// Handles what happens when a new session is created from the Hub
// The hub passes us a session ID. We want to encode that Session ID so that the client gets an encoded session id
// resp should be the original response from the hub
// Returns a modified http response, and an error if applicable
func handleNewSession(req *Request, resp *http.Response) error {
	// Create a new session based off the response
	if err := req.Session.InitSessionFromResponse(resp); err != nil {
		return err
	}

	// instrument the new session to Cloudwatch.
	go func() {
		_, err := WriteMetricNewSession(req)
		if err != nil { // log but keep going, this metric is non-critical
			req.AppConfig.Logger.Errorf("encountered error writing metric new session: %v", err)
		}
	}()

	// Modify the body with the encoded session id
	return replaceBodyWithExternalID(req.Session, resp)
}

// Takes the body that is returned by the Hub with the session ID, and replaces it with the SID to share externally
// This is so that clients will use the External ID in future communication to Grid Router
func replaceBodyWithExternalID(session *Session, resp *http.Response) error {
	body, err := ioutil.ReadAll(resp.Body)
	resp.Body.Close() // We no longer need that body, we'll replace it
	if err != nil {
		return err
	}

	// Create a new body that has the session id for the client
	newBody, err := getNewSessionBodyForClient(session, &body)
	if err != nil {
		return err
	}

	// Put the new body back into the response for the client
	resp.Body = ioutil.NopCloser(bytes.NewReader(newBody))
	resp.ContentLength = int64(len(newBody))
	resp.Header.Set("Content-Length", strconv.Itoa(len(newBody)))
	return nil
}

// This will get the new session body for the client, which will replace the session id
func getNewSessionBodyForClient(session *Session, body *[]byte) ([]byte, error) {
	if session == nil || body == nil {
		return nil, errors.New("session or body was nil")
	}

	if session.ExternalID == nil || session.InternalID == nil {
		return nil, errors.New("the Internal or External Session ID were blank")
	}

	// Replace the Internal SID with the External SID, since we're giving it to the client
	bodyStr := string(*body)
	newBody := strings.Replace(bodyStr, *session.InternalID, *session.ExternalID, -1)
	return []byte(newBody), nil
}

// Writes a Metric to Cloudwatch for when a new session is created
func WriteMetricNewSession(req *Request) (*cloudwatch.PutMetricDataOutput, error) {
	// protect against nil lookups
	if req == nil || req.AppConfig == nil || req.AppConfig.Instrumentor == nil || req.AppConfig.Clock == nil {
		return nil, errors.New("required request instrumentor or clock is missing")
	}

	metric := &cloudwatch.PutMetricDataInput{
		Namespace: aws.String("CBG"),
		MetricData: []*cloudwatch.MetricDatum{
			{
				MetricName: aws.String("NewSession"),
				Timestamp: aws.Time(req.AppConfig.Clock.Now()),
				Value: aws.Float64(1.0),
				Unit: aws.String(cloudwatch.StandardUnitCount),
				Dimensions: []*cloudwatch.Dimension{
					{Name: aws.String("AutoScalingGroupName"), Value: aws.String(req.AppConfig.Instrumentor.AutoScalingGroupName)},
				},
			},
		},
	}

	req.AppConfig.Logger.Debugf("writeMetricNewSession() Sending Metric to Cloudwatch: %s", metric.String())
	result, err := req.AppConfig.Instrumentor.PutMetricData(metric)
	req.AppConfig.Logger.Debugf("writeMetricNewSession() Cloudwatch Metric Result: %s", result.String())
	return result, err
}
