// Package prox is responsible for the internals of the Proxy Server
package session

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"errors"
	"net"
	"net/http"
	"net/http/httputil"
	"net/url"
	"regexp"
	"strconv"
	"time"
)

const (
	errorUnknownRoute = "an unknown route was passed to the handler"
	errorHandlingRequest = "encountered an error handling your request"
)

// Creates a new Session Proxy
// This coordinates communication between this application and the Hub
func NewProxy(appConfig *config.Config, registry *hub_registry.RedisRegistry) *httputil.ReverseProxy {
	proxy := httputil.NewSingleHostReverseProxy(&url.URL{})
	proxy.Transport = newTransport(appConfig, registry)

	proxy.ErrorHandler = func(w http.ResponseWriter, r *http.Request, err error) {
		appConfig.Logger.Errorf("encountered error in roundtripper: %v", err)

		// TODO - Do not use a generic error
		http.Error(w, errorHandlingRequest, http.StatusInternalServerError)
	}
	return proxy
}

type transport struct {
	RoundTripper    http.RoundTripper
	appConfig       *config.Config
	Registry        *hub_registry.RedisRegistry
}

// Creates a new Transport object used for the proxy
func newTransport(appConfig *config.Config, registry *hub_registry.RedisRegistry) *transport {
	return &transport{
		RoundTripper:   newRoundtripTransport(),
		appConfig:      appConfig,
		Registry:       registry,
	}
}

// Creates a new Roundtripper Transport Object
func newRoundtripTransport() *http.Transport {
	return &http.Transport{
		Proxy: http.ProxyFromEnvironment,
		DialContext: (&net.Dialer{
			Timeout:   30 * time.Second,
			KeepAlive: 30 * time.Second,
			DualStack: true,
		}).DialContext,
		MaxIdleConns:          100,
		IdleConnTimeout:       305 * time.Second,
		TLSHandshakeTimeout:   30 * time.Second,
		ExpectContinueTimeout: 30 * time.Second,
		ResponseHeaderTimeout: 305 * time.Second, // Wait 305 seconds for a response. Grid takes awhile to perform actions
	}
}

// Passes the Client's HTTP Request to the Hub
// Contains Before/After hooks to properly handle the request
func (t *transport) RoundTrip(req *http.Request) (*http.Response, error) {
	gridRequest, err := NewRequest(req, t.Registry, t.appConfig)
	if err != nil {
		return nil, err
	}

	// Run Before Hooks
	if err := t.Before(&gridRequest); err != nil {
		return nil, err
	}

	// Send the request off to the hub
	startTime := time.Now()
	t.appConfig.Logger.Infof("-> %s", req.URL.String())
	resp, err := t.RoundTripper.RoundTrip(req)
	if err != nil {
		return nil, err
	}
	reqDuration := time.Now().Sub(startTime)
	t.appConfig.Logger.Infof("<- [%d] | %s | %s", resp.StatusCode, reqDuration, req.URL.Path)

	// Run After Hooks
	if err := t.After(&gridRequest, resp); err != nil {
		return resp, err
	}

	return resp, nil
}

// Prepares the request to be sent to the proper Hub
// Also converts Session ID from External to Internal so that the Hub can understand it
func (t *transport) Before(gridRequest *Request) error {
	if gridRequest.IsNewSessionRequest() {
		if err := handleNewSessionBefore(gridRequest); err != nil {
			return err
		}
	} else if gridRequest.IsExistingSessionRequest() {
		// Replace the path with the internal session id, that we can send to hub
		hubPath, err := gridRequest.HubPath()
		if err != nil {
			return err
		}

		// Set the request to that path
		gridRequest.RawRequest.URL.Path = hubPath
	} else { // if not a NewSession or not an existing session
		return errors.New(errorUnknownRoute)
	}

	// Replace the Request URL's Host with the Hub's Host
	host, err := gridRequest.HubHost()
	if err != nil {
		return err
	}
	gridRequest.RawRequest.URL.Scheme = "http"
	gridRequest.RawRequest.URL.Host = host
	return nil
}



// Logic that should be made after a response is made from the hub, but before it's handed back to the client
// This includes logic like taking an Internal Session ID and replacing it with an External Session ID for the client
// Req is the HTTP Request that was sent
// Resp is the HTTP Response that the target responded with
func (t *transport) After(gridReq *Request, resp *http.Response) error {
	// If the response was an error, don't do anything with it. Preserve the original error to give to the client.
	if isHTTPError(resp) {
		t.appConfig.Logger.Errorf("Encountered an HTTP Error. Status code: %d. URL: %s. Returning original response.",
			resp.StatusCode, resp.Request.URL)
		return nil
	}

	// If the is a new session, run the New Session handler
	if gridReq.IsNewSessionRequest() {
		return handleNewSession(gridReq, resp)
	} else if gridReq.IsDeleteSessionRequest() {
		t.appConfig.Logger.Debugf("Adding 1 FreeSlot to hub: %s", gridReq.Session.Hub.ID)
		gridReq.Session.Hub.SlotCounts.Free += 1
		err := t.Registry.SaveHub(gridReq.Session.Hub)
		if err != nil {
			t.appConfig.Logger.Warnf("Encountered error saving hub to redis: %v", err)
			// swallow it, do not return an error back to the user for this
		}
	}

	return nil
}

// isHTTPError determines if the response code is of 4xx or 5xx
// resp is the HTTP Response
// return a boolean of if the response code is of 4xx or 5xx
func isHTTPError(resp *http.Response) bool {
	pattern := "^(4|5)" // String starts with 4 or 5
	r := regexp.MustCompile(pattern)
	statusCode := strconv.Itoa(resp.StatusCode) // Convert int status code to string
	return r.MatchString(statusCode)
}
