package session

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor"
	"code.justin.tv/qe/grid_router/src/pkg/instrumentor/mocks"
	"encoding/json"
	"fmt"
	"github.com/alicebob/miniredis/v2"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/go-redis/redis"
	"github.com/jonboulle/clockwork"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"io/ioutil"
	"log"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"
	"text/template"
	"time"
)

func TestNewProxy(t *testing.T) {
	registry := &hub_registry.RedisRegistry{}
	appConfig := config.NewMock()

	t.Run("creates a new roundtrip proxy", func (t *testing.T) {
		proxy := NewProxy(appConfig, registry)
		assert.NotNil(t, proxy) // unfortunately not a lot to test - the fields aren't exposed
	})
}

func TestTransport_RoundTrip(t *testing.T) {
	hubServerRequests := 0
	internalSessionID := "testSID" // used as the session id that is mocked back as created
	var actualURLPath string // declare outside of the hub server so we can access it

	// Set up a mock hub server that pretends to create a session
	hubServerNewSession := createMockHubServer(&hubServerRequests, &actualURLPath, nil, internalSessionID)
	defer hubServerNewSession.Close()

	// Additional Setup, run a redis server, set up the registry
	s, err := miniredis.Run()
	defer s.Close()
	require.NoError(t, err)

	redisClient := redis.NewClient(&redis.Options{
		Addr: s.Addr(),
	})
	defer redisClient.Close()

	// Config & Metric Setup
	metricWriterMock := &mocks.MetricWriter{}
	loggerMock, _ := test.NewNullLogger()
	appConfig := config.NewMock()
	appConfig.Instrumentor = &instrumentor.Instrumentor{
		MetricClient: metricWriterMock,
		Logger: loggerMock,
	}

	// Create a registry
	reg := hub_registry.NewRegistry(nil, redisClient, appConfig, time.Hour)

	// Assign a fake clock to the registry so that we can control it
	mockClock := clockwork.NewFakeClock()
	appConfig.Clock = mockClock

	hubServerUrl, err := url.Parse(hubServerNewSession.URL)
	require.NoError(t, err)

	sessionProxy := NewProxy(appConfig, reg)

	// Create a mock hub with the host and port from the test server above
	mockHub := hub_registry.Hub{
		ID: "i-1234",
		IP: hubServerUrl.Hostname(),
		Port: hubServerUrl.Port(),
		Healthy: true,
		Paused: false,
		SlotCounts: hub_registry.SlotCounts{
			Free: 1,
			Total: 1,
		},
	}

	externalSid := fmt.Sprintf("%s_%s", internalSessionID, mockHub.ID)
	existingSessionURL := fmt.Sprintf("/wd/hub/session/%s/url", externalSid)

	t.Run("Create New Session", func(t *testing.T) {
		t.Run("when there is no hub", func (t *testing.T) {
			actualURLPath = "" // clear out old data
			hubServerRequests = 0 // clear out old data
			redisClient.FlushAll() // clear out old data

			// Prepare the request to create a new session
			req, err := http.NewRequest(http.MethodPost, sessionRoute, nil)
			require.NoError(t, err)
			w := httptest.NewRecorder()

			// Make the request to the handler
			assert.Equal(t, 0, hubServerRequests, "should have 0 requests before the request")

			// set up an async go routine to advance the clock, simulating a minute has passed
			// this is to speed up the test so that it doesn't wait the full minute
			go func (c clockwork.FakeClock) {
				time.Sleep(time.Millisecond / 2) // give the test a chance to call the Handle function before advancing
				c.Advance(time.Minute)
			}(mockClock)

			sessionProxy.ServeHTTP(w, req)
			assert.Equal(t, 0, hubServerRequests, "request should have never made it since there is no hub")

			resultBody, err := ioutil.ReadAll(w.Result().Body)
			defer w.Result().Body.Close()
			require.NoError(t, err)

			// TODO - Make a 503 when we have unique error handling
			assert.Equal(t, http.StatusInternalServerError, w.Result().StatusCode)
			assert.Equal(t, errorHandlingRequest + "\n", string(resultBody))
		})

		t.Run("when there is one hub", func (t *testing.T) {
			actualURLPath = "" // clear out old data
			hubServerRequests = 0 // clear out old data
			redisClient.FlushAll() // clear out old data

			// Handle a mocked call to PutMetricData
			metricWriterMock.On("PutMetricData", mock.Anything).Return(&cloudwatch.PutMetricDataOutput{}, nil).Once()

			// Load a hub in
			err = reg.SaveHub(&mockHub) // save that hub into redis and the registry now
			require.NoError(t, err)

			// Prepare the request to create a new session
			req, err := http.NewRequest(http.MethodPost, sessionRoute, nil)
			require.NoError(t, err)
			w := httptest.NewRecorder()

			// Make the request to the handler
			assert.Equal(t, 0, hubServerRequests, "should have 0 requests before the request")
			sessionProxy.ServeHTTP(w, req)
			assert.Equal(t, 1, hubServerRequests, "should have 1 request after")

			resp := w.Result()
			assert.Equal(t, 200, resp.StatusCode)
			body, err := ioutil.ReadAll(resp.Body)
			require.NoError(t, err)

			// A struct to store the response
			type BodyJSON struct {
				Value struct {
					SessionID string
				}
			}
			bodyResponse := BodyJSON{}
			err = json.Unmarshal(body, &bodyResponse)
			require.NoError(t, err)
			assert.Equal(t, externalSid, bodyResponse.Value.SessionID) // ensure the session ID is returned as external

			t.Run("registry slot count free should have been decremented", func (t *testing.T) {
				hubResult, err := reg.GetHubById(mockHub.ID)
				require.NoError(t, err)
				require.NotNil(t, hubResult)
				assert.Equal(t, mockHub.SlotCounts.Free - 1, hubResult.SlotCounts.Free)
			})

			t.Run("NewSession metric was written", func (t *testing.T) {
				metricWriterMock.AssertNumberOfCalls(t, "PutMetricData", 1)
			})
		})
	})

	t.Run("Existing session gets routed with the proper session id", func(t *testing.T) {
		actualURLPath = "" // clear out old data
		hubServerRequests = 0 // clear out old data
		redisClient.FlushAll() // clear out old data

		// Load a hub in
		err = reg.SaveHub(&mockHub) // save that hub into redis and the registry now
		require.NoError(t, err)

		// Create the request
		req, err := http.NewRequest(http.MethodGet, existingSessionURL, nil)
		require.NoError(t, err)
		w := httptest.NewRecorder()

		// Make the request to the handler
		assert.Equal(t, 0, hubServerRequests, "should have 0 requests before the request")
		sessionProxy.ServeHTTP(w, req)
		assert.Equal(t, 1, hubServerRequests, "should have 1 request after")

		resp := w.Result()
		assert.Equal(t, 200, resp.StatusCode)
		assert.Equal(t, fmt.Sprintf("%s/%s/url", sessionRoute, internalSessionID), actualURLPath, "the path from the hub should have used the internal id")

		t.Run("registry slot count free should remain the same", func (t *testing.T) {
			hubResult, err := reg.GetHubById(mockHub.ID)
			require.NoError(t, err)
			require.NotNil(t, hubResult)
			assert.Equal(t, mockHub.SlotCounts.Free, hubResult.SlotCounts.Free)
		})
	})

	t.Run("Test Delete Session Request Gets Routed", func(t *testing.T) {
		actualURLPath = "" // clear out old data
		hubServerRequests = 0 // clear out old data
		redisClient.FlushAll() // clear out old data

		// Load a hub in
		mockHubCopy := *&mockHub
		mockHubCopy.SlotCounts.Free = 0
		err = reg.SaveHub(&mockHubCopy) // save that hub into redis and the registry now
		require.NoError(t, err)

		// Create the request
		deletePath := fmt.Sprintf("%s/%s", sessionRoute ,externalSid)
		req, err := http.NewRequest(http.MethodDelete, deletePath, nil)
		require.NoError(t, err)
		w := httptest.NewRecorder()

		// Make the request to the handler
		assert.Equal(t, 0, hubServerRequests, "should have 0 requests before the request")
		sessionProxy.ServeHTTP(w, req)
		assert.Equal(t, 1, hubServerRequests, "should have 1 request after")

		resp := w.Result()
		assert.Equal(t, 200, resp.StatusCode)
		assert.Equal(t, fmt.Sprintf("%s/%s", sessionRoute, internalSessionID), actualURLPath, "the path from the hub should have used the internal id")

		t.Run("registry slot count free should have been incremented", func (t *testing.T) {
			hubResult, err := reg.GetHubById(mockHub.ID)
			require.NoError(t, err)
			require.NotNil(t, hubResult)
			assert.Equal(t, mockHubCopy.SlotCounts.Free + 1, hubResult.SlotCounts.Free)
		})
	})

	t.Run("unknown route", func (t *testing.T) {
		t.Run("blank", func (t *testing.T) {
			req, err := http.NewRequest(http.MethodGet, "", nil)
			require.NoError(t, err)
			w := httptest.NewRecorder()

			sessionProxy.ServeHTTP(w, req)

			resultBody, err := ioutil.ReadAll(w.Result().Body)
			defer w.Result().Body.Close()
			require.NoError(t, err)

			// TODO - it should be a 404 when we have unique error handling
			//assert.Equal(t, http.StatusNotFound, w.Result().StatusCode)
			//assert.Equal(t, errorUnknownRoute + "\n", string(resultBody))
			assert.Equal(t, http.StatusInternalServerError, w.Result().StatusCode)
			assert.Equal(t, errorHandlingRequest + "\n", string(resultBody))
		})

		t.Run("unknown", func (t *testing.T) {
			req, err := http.NewRequest(http.MethodGet, "/unknown", nil)
			require.NoError(t, err)
			w := httptest.NewRecorder()

			sessionProxy.ServeHTTP(w, req)

			resultBody, err := ioutil.ReadAll(w.Result().Body)
			defer w.Result().Body.Close()
			require.NoError(t, err)

			// TODO - it should be a 404 when we have unique error handling
			//assert.Equal(t, http.StatusNotFound, w.Result().StatusCode)
			//assert.Equal(t, errorUnknownRoute + "\n", string(resultBody))
			assert.Equal(t, http.StatusInternalServerError, w.Result().StatusCode)
			assert.Equal(t, errorHandlingRequest + "\n", string(resultBody))
		})
	})
}

func TestTransport_After(t *testing.T) {
	t.Run("on an http error", func (t *testing.T) {
		testUrlPath := "http://mytest.com"
		responseCode := 500
		expectedError := fmt.Sprintf("Encountered an HTTP Error. Status code: %d. URL: %s. Returning original response.",
			responseCode, testUrlPath)

		// Create the URL
		testUrl, err := url.Parse(testUrlPath)
		require.NoError(t, err)

		// Create the mock response that would be returned
		mockRespErr := &http.Response{
			StatusCode: responseCode,
			Request: &http.Request{URL: testUrl},
		}

		// Create a Logger and Hook so we can inspect the error
		logger, hook := test.NewNullLogger()

		mockRequest := &Request{}

		config := config.NewMock()
		config.Logger = logger

		// Create the Transport Object
		transporter := transport{
			appConfig: config,
		}

		assert.Len(t, hook.Entries, 0)
		err = transporter.After(mockRequest, mockRespErr)
		assert.NoError(t, err, "error should have been swallowed")
		require.Len(t, hook.Entries, 1)

		// ensure the error was logged
		assert.Equal(t, expectedError,
			hook.Entries[0].Message)
		assert.Equal(t, logrus.ErrorLevel, hook.Entries[0].Level)
	})
}

// Creates a Hub Server that will act as a mock
// Most parameters are a pointer, as they will allow the Hub Server to modify them to help test and extract data
// If there is a parameter you are not testing, provide nil and it will not be used
func createMockHubServer(hubServerRequests *int, actualURLPath *string, header *string, internalSessionID string) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if hubServerRequests != nil { *hubServerRequests += 1 }
		if actualURLPath     != nil { *actualURLPath = r.URL.RequestURI() }
		if header            != nil { *header = r.Header.Get("Accept-Encoding") }

		w.Header().Set("Content-Type", "application/json")
		if r.Method == http.MethodPost { // New Session Response - Mock a response back
			writeMockCreateSessionResponse(internalSessionID, w)
		}
	}))
}

// Fetches a CreateSessionResponse Template, replaces a mock Session ID, and then writes it to the HTTP Response
func writeMockCreateSessionResponse(sessionID string, writer http.ResponseWriter) {
	// Create the template object
	t, err := template.ParseFiles("./testdata/mockHubResponse.json")
	if err != nil {
		log.Fatalf("encountered error reading file: %v", err)
	}

	// Load the session ID into the template
	err = t.Execute(writer, sessionID) // write with the updated template
	if err != nil {
		log.Fatalf("executing template: %v", err)
	}
}
