package session

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"errors"
	"fmt"
	"net/http"
	"regexp"
	"strings"
	"time"
)

// Represents an HTTP Request meant for Selenium Grid
type Request struct {
	RawRequest *http.Request
	Session    *Session
	Registry   hub_registry.Registry
	AppConfig  *config.Config
}

// Creates a new Grid Request object from an http request
func NewRequest(rawRequest *http.Request, hubRegistry hub_registry.Registry, appConfig *config.Config) (Request, error) {
	r := Request{RawRequest: rawRequest, Registry: hubRegistry, AppConfig: appConfig}

	if r.IsExistingSessionRequest() {
		ses, err := r.NewSession()
		if err != nil { // If no error
			return r, err
		}
		r.Session = ses
	}

	return r, nil
}

func (req Request) IsExistingSessionRequest() bool {
	path := req.RawRequest.URL.Path
	pattern := getSessionPathRegex()
	re := regexp.MustCompile(pattern)
	return re.Match([]byte(path))
}

// Determines if a request is a New Selenium Session Request
func (req Request) IsNewSessionRequest() bool {
	return strings.ToLower(req.RawRequest.Method) == "post" && strings.ToLower(req.RawRequest.URL.RequestURI()) == "/wd/hub/session"
}

// Returns if a request is a delete session request
func (req Request) IsDeleteSessionRequest() bool {
	if strings.ToLower(req.RawRequest.Method) != "delete" {
		return false
	}
	// Look for a request like DELETE /wd/hub/SESSION_ID
	// Allow a trailing slash, like /wd/hub/SESSION_ID/
	// Do not allow for anything after a trailing slash, as that may be a request to delete an element
	pattern := "\\/wd\\/hub\\/session\\/[a-zA-Z0-9_-]+\\/?$"
	r := regexp.MustCompile(pattern)
	return r.Match([]byte(strings.ToLower(req.RawRequest.URL.RequestURI())))
}

// getSessionPathRegex will return a regular expression for searching session paths
// Pattern should look for /session/SESSION_ID/anything else
// Delete however will have just /session/SESSION_ID, so make the ending / optional.
// 2nd index will be the Session ID
// returns a string containing a regular expression for Session Paths
func getSessionPathRegex() string {
	return "^.*wd\\/hub\\/session\\/([a-zA-Z0-9_-]+)(\\/.*|$)"
}

// Parses the SessionID from an existing request URL
// When a client wants to send a session command, it's passed as a url like /session/SESSION_ID/command
// This function will parse the Session ID from that path
// path should be the request path
// returns a string containing the Session ID
func (req Request) GetSessionIDFromPath() (string, error) {
	path := req.RawRequest.URL.Path
	pattern := getSessionPathRegex()

	r, err := regexp.Compile(pattern)
	if err != nil {
		return "", err
	}

	subMatch := r.FindStringSubmatch(path)
	// The 2nd index should be the session id. Make sure there's at least 2
	if subMatch == nil || len(subMatch) < 2 {
		req.AppConfig.Logger.Warn("GetSessionIDFromPath(): Unexpected error finding the session id")
		return "", errors.New("Could not parse the path: " + path)
	}

	return subMatch[1], nil
}

// Returns the path that should be used when communicating with the hub
// This handles taking the External ID and transforming it to the Internal ID
func (req Request) HubPath() (string, error) {
	if req.Session == nil {
		return "", errors.New("session had a nil value that was required")
	}

	if req.Session.InternalID == nil || req.Session.ExternalID == nil {
		return "", fmt.Errorf("session had a nil session id that was required." +
			" Internal ID: '%v' External ID: '%v' Request Path: '%s'",
			req.Session.InternalID, req.Session.ExternalID, req.RawRequest.RequestURI)
	}

	if req.RawRequest == nil || req.RawRequest.URL == nil {
		return "", errors.New("request had a nil raw request that was required")
	}

	// Replace All References of the External ID with the Internal ID
	newPath := strings.Replace(req.RawRequest.URL.Path, *req.Session.ExternalID, *req.Session.InternalID, -1)
	return newPath, nil
}

// Returns the hub host for the current session, including the port
// Example: 127.0.0.1:4444
func (req Request) HubHost() (string, error) {
	if req.Session == nil || req.Session.Hub == nil || req.Session.Hub.Port == "" {
		return "", errors.New("session, hub or port were nil")
	}

	var host string

	// Prefer using host name. If not provided, use IP
	if req.Session.Hub.Hostname != "" {
		host = req.Session.Hub.Hostname
	} else if req.Session.Hub.IP != "" {
		host = req.Session.Hub.IP
	} else {
		return "", errors.New("a hostname or ip were not provided for the hub")
	}

	return fmt.Sprintf("%s:%s", host, req.Session.Hub.Port), nil
}

// Creates and populates a session object from a Grid Request
func (req Request) NewSession() (*Session, error) {
	var s Session

	extID, err := req.GetExternalID()
	if err != nil {
		return nil, err
	}
	intId, err := req.GetInternalID()
	if err != nil {
		return nil, err
	}

	// Get the Hub Information
	hubId, err := req.GetHubID()
	if err != nil {
		return nil, err
	}

	// Get the Hub from the registry
	// Uses a Timeout to keep retrying, in the event Redis returns a connection error.
	// This way, if Redis goes down, we can keep trying without simply returning a 500
	var hub *hub_registry.Hub
	timeout := req.AppConfig.Clock.Now().Add(req.AppConfig.ForwardRequestTimeout)
	for req.AppConfig.Clock.Now().Before(timeout) {
		hub, err = req.Registry.GetHubById(hubId)
		if err == nil && hub != nil {
			break
		}
		req.AppConfig.Logger.Warnf("Encountered error while getting hub: %v. Retrying", err)
		req.AppConfig.Clock.Sleep(time.Millisecond * 100)
	}

	if err != nil {
		return &s, fmt.Errorf("timed out trying to fetch hub: %v", err)
	}
	if hub == nil {
		return &s, errors.New("error finding hub")
	}

	// Populate the session object
	s = Session{InternalID: &intId, ExternalID: &extID, Hub: hub, AppConfig: req.AppConfig} // Create the object
	return &s, nil
}

// Returns the Internal ID that should be used for the Hub
// External Session IDs are in the format of <internal>_<custom_external>. This will strip the _<custom_external>
// Example: hr8en365_i-3896, where hr8en365 is internal
func (req Request) GetInternalID() (string, error) {
	sessionID, err := req.GetSessionIDFromPath()
	if err != nil {
		return "", err
	}

	hubId, err := req.GetHubID()
	if err != nil {
		return "", err
	}
	// The format of the external custom id that was appended.
	extSid := fmt.Sprintf("_%s", hubId)

	// Remove the external id encoding
	dSid := strings.Replace(sessionID, extSid, "", 1)
	req.AppConfig.Logger.Debugf("Decoding %s to %s", sessionID, dSid)

	return dSid, nil
}

// Returns the External ID that should be used for the client
func (req Request) GetExternalID() (string, error) {
	sessionID, err := req.GetSessionIDFromPath()
	if err != nil {
		return "", err
	}

	return sessionID, nil
}

// Returns the Hub Identifier from the Session ID
func (req Request) GetHubID() (string, error) {
	sessionID, err := req.GetSessionIDFromPath()
	if err != nil {
		return "", err
	}

	// Regex that looks for <internal_sid>_<hub_id>, and returns the hub_id
	// example: h7359nf_i-3j59 would return i-3j59
	pattern := "\\S+_(i-\\w+)$"
	r, err := regexp.Compile(pattern)
	if err != nil {
		return "", err
	}

	// Find the match from the session id using the regex
	subMatch := r.FindStringSubmatch(sessionID)
	// The 2nd index should be the session id. Make sure there's at least 2
	if subMatch == nil || len(subMatch) < 2 {
		req.AppConfig.Logger.Errorf("GetHubID(): Unexpected error finding the hub id")
		return "", errors.New("Could not parse the hub id from sid: " + sessionID)
	}

	return subMatch[1], nil
}
