package session

import (
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry/mocks"
	hub_registrytest "code.justin.tv/qe/grid_router/src/pkg/hub_registry/test"
	"errors"
	"fmt"
	"github.com/alicebob/miniredis/v2"
	"github.com/elliotchance/redismock/v6"
	"github.com/go-redis/redis"
	"github.com/jonboulle/clockwork"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"net/http"
	"testing"
	"time"
)

var intSessionID = "test-sid"
var hubID = "i-1234"
var extSessionID = fmt.Sprintf("%s_%s", intSessionID, hubID)
var sessionRoute = "/wd/hub/session"
var existingRouteWithSid = fmt.Sprintf("%s/%s", sessionRoute, extSessionID)
var existingUrl = fmt.Sprintf("http://myserver.com%s", existingRouteWithSid)

func TestNewRequest(t *testing.T) {
	redisClient := newTestRedis()

	reg := &hub_registry.RedisRegistry{
		DBClient:  redisClient,
		AppConfig: config.NewMock(),
	}

	hub := &hub_registry.Hub{
		ID: "i-1234",
		IP: "1.2.3.4",
		Port: "5678",
	}
	err := reg.SaveHub(hub)
	if err != nil {
		t.Fatalf("error saving hub into registry: %v", err)
	}
	appConfig := config.NewMock()

	t.Run("Returns a new request", func(t *testing.T) {
		rawRequest, err := http.NewRequest(http.MethodGet, "http://mytest.com", nil)
		require.NoError(t, err)

		gridRequest, err := NewRequest(rawRequest, reg, appConfig)
		assert.NoError(t, err)
		assert.Equal(t, reg, gridRequest.Registry, "should populate the registry")
		assert.Equal(t, appConfig, gridRequest.AppConfig, "should populate the app config")
		assert.Equal(t, rawRequest, gridRequest.RawRequest, "should populate the raw request")
	})

	t.Run("populates session if existing request", func(t *testing.T) {
		url := fmt.Sprintf("http://server.com/wd/hub/session/%s", extSessionID)
		rawRequest, err := http.NewRequest(http.MethodGet, url, nil)
		require.NoError(t, err)

		gridRequest, err := NewRequest(rawRequest, reg, appConfig)
		assert.NoError(t, err)
		assert.NotNil(t, gridRequest.Session)
		assert.NotNil(t, gridRequest.Session.InternalID, "should have an internal id")
		assert.NotNil(t, gridRequest.Session.ExternalID, "should have an external id")
		assert.Equal(t, intSessionID, *gridRequest.Session.InternalID)
		assert.Equal(t, extSessionID, *gridRequest.Session.ExternalID)
	})
}

func TestRequest_NewSession(t *testing.T) {
	mockClock := clockwork.NewFakeClock()
	reg := &mocks.Registry{}
	config := config.NewMock()
	config.Clock = mockClock

	url := fmt.Sprintf("http://server.com/wd/hub/session/%s", extSessionID)
	rawRequest, err := http.NewRequest(http.MethodGet, url, nil)
	require.NoError(t, err)

	request := Request{
		RawRequest: rawRequest,
		Session:    nil,
		Registry:   reg,
		AppConfig:  config,
	}

	t.Run("returns immediately if GetHubById is functioning", func (t *testing.T) {
		hubToRespond := hub_registrytest.CreateMockTestHub()
		reg.On("GetHubById", hubID).Return(hubToRespond, nil).Once()

		resp, err := request.NewSession()
		assert.NoError(t, err)
		require.NotNil(t, resp)
		hub_registrytest.AssertEqualHub(t, hubToRespond, resp.Hub)
	})

	t.Run("returns if GetHubById returns results prior to timeout", func (t *testing.T) {
		hubToRespond := hub_registrytest.CreateMockTestHub()
		reg.Calls = nil // reset the calls, we assert the number of calls later
		reg.On("GetHubById", hubID).Return(nil, errors.New("testing")).Times(2)

		// set up an async go routine to advance the clock, simulating time passing, and eventually returning
		go func (c clockwork.FakeClock) {
			time.Sleep(time.Millisecond)
			c.Advance(request.AppConfig.ForwardRequestTimeout / 2) // Time pass, still erroring
			time.Sleep(time.Millisecond)
			reg.On("GetHubById", hubID).Return(hubToRespond, nil).Once()
			c.Advance(time.Second) // Time pass, now responding with a hub
		}(mockClock)

		resp, err := request.NewSession()
		assert.NoError(t, err)
		require.NotNil(t, resp)
		hub_registrytest.AssertEqualHub(t, hubToRespond, resp.Hub)
		reg.AssertNumberOfCalls(t, "GetHubById", 3) // should have retried
	})

	t.Run("returns error if GetHubById errors for longer than timeout", func (t *testing.T) {
		reg.On("GetHubById", hubID).Return(nil, errors.New("test"))

		// set up an async go routine to advance the clock, simulating a minute has passed
		// this is to speed up the test so that it doesn't wait the full minute
		go func (c clockwork.FakeClock) {
			time.Sleep(time.Millisecond) // give the test a chance to call before advancing
			c.Advance(request.AppConfig.ForwardRequestTimeout)
		}(mockClock)

		_, err = request.NewSession()
		assert.Error(t, err)
	})

	t.Run("returns error if GetHubById nil hub for longer than timeout", func (t *testing.T) {
		reg.On("GetHubById", hubID).Return(nil, nil)
		reg.Calls = nil

		// set up an async go routine to advance the clock, simulating a minute has passed
		go func (c clockwork.FakeClock) {
			time.Sleep(time.Millisecond)
			c.Advance(time.Second) // call +1
			time.Sleep(time.Millisecond)
			c.Advance(request.AppConfig.ForwardRequestTimeout) // finish and timeout
		}(mockClock)

		_, err = request.NewSession()
		assert.Error(t, err)
		reg.AssertNumberOfCalls(t, "GetHubById", 2) // verify it retried
	})
}

func TestRequest_IsNewSessionRequest(t *testing.T) {
	t.Run("http methods", func (t *testing.T) {
		url := fmt.Sprintf("http://server.com%s", sessionRoute)

		var methodTest = []struct {
			in string
			out bool
		}{
			{http.MethodGet, false},
			{http.MethodPost, true},
			{http.MethodPut, false},
			{http.MethodDelete, false},
		}
		for _, tt := range methodTest {
			t.Run(tt.in, func(t *testing.T) {
				req, err := http.NewRequest(tt.in, url, nil)
				require.NoError(t, err)

				gridReq := Request{
					RawRequest: req,
				}

				assert.Equal(t, tt.out, gridReq.IsNewSessionRequest())
			})
		}
	})

	t.Run("paths", func (t *testing.T) {
		var pathTest = []struct {
			in string
			out bool
		}{
			{sessionRoute, true},
			{existingRouteWithSid, false},
			{"/not/the/route", false},
		}
		for _, tt := range pathTest {
			t.Run(tt.in, func(t *testing.T) {
				req, err := http.NewRequest(http.MethodPost, tt.in, nil)
				require.NoError(t, err)

				gridReq := Request{
					RawRequest: req,
				}

				assert.Equal(t, tt.out, gridReq.IsNewSessionRequest())
			})
		}
	})
}

func TestRequest_IsExistingSessionRequest(t *testing.T) {
	route := fmt.Sprintf("%s/url", existingRouteWithSid)

	t.Run("http methods", func (t *testing.T) {
		var methodTest = []struct {
			in string
			out bool
		}{
			{http.MethodGet, true},
			{http.MethodPost, true},
			{http.MethodPut, true},
			{http.MethodDelete, true},
		}
		for _, tt := range methodTest {
			t.Run(tt.in, func(t *testing.T) {
				req, err := http.NewRequest(tt.in, existingUrl, nil)
				require.NoError(t, err)

				gridReq := Request{
					RawRequest: req,
				}

				assert.Equal(t, tt.out, gridReq.IsExistingSessionRequest())
			})
		}
	})

	t.Run("paths", func (t *testing.T) {
		var pathTest = []struct {
			in string
			out bool
		}{
			{route, true},
			{"/not/the/route", false},
			{"/wd/hub/session", false},
		}
		for _, tt := range pathTest {
			t.Run(tt.in, func(t *testing.T) {
				req, err := http.NewRequest(http.MethodPost, tt.in, nil)
				require.NoError(t, err)

				gridReq := Request{
					RawRequest: req,
				}

				assert.Equal(t, tt.out, gridReq.IsExistingSessionRequest())
			})
		}
	})
}

func TestRequest_IsDeleteSessionRequest(t *testing.T) {
	t.Run("http methods", func (t *testing.T) {
		var methodTest = []struct {
			in string
			out bool
		}{
			{http.MethodGet, false},
			{http.MethodPost, false},
			{http.MethodPut, false},
			{http.MethodDelete, true},
		}
		for _, tt := range methodTest {
			t.Run(tt.in, func(t *testing.T) {
				req, err := http.NewRequest(tt.in, existingUrl, nil)
				require.NoError(t, err)

				gridReq := Request{
					RawRequest: req,
				}

				assert.Equal(t, tt.out, gridReq.IsDeleteSessionRequest())
			})
		}
	})

	t.Run("paths", func (t *testing.T) {
		var pathTest = []struct {
			in string
			out bool
		}{
			{existingUrl, true},
			{"/not/the/route", false},
			{"/wd/hub/session", false},
			{"/wd/hub/session/i-1234/element", false}, // deleting an element/etc. shouldn't be a session delete
		}
		for _, tt := range pathTest {
			t.Run(tt.in, func(t *testing.T) {
				req, err := http.NewRequest(http.MethodDelete, tt.in, nil)
				require.NoError(t, err)

				gridReq := Request{
					RawRequest: req,
				}

				assert.Equal(t, tt.out, gridReq.IsDeleteSessionRequest())
			})
		}
	})
}

func TestRequest_GetSessionIDFromPath(t *testing.T) {
	t.Run("valid path", func(t *testing.T) {
		req, err := http.NewRequest(http.MethodGet, existingUrl, nil)
		require.NoError(t, err)

		gridReq := Request{
			RawRequest: req,
		}

		result, err := gridReq.GetSessionIDFromPath()
		assert.NoError(t, err)
		assert.Equal(t, extSessionID, result)
	})

	t.Run("invalid path", func(t *testing.T) {
		req, err := http.NewRequest(http.MethodGet, "http://myserver.com/grid/console", nil)
		require.NoError(t, err)

		gridReq := Request{
			RawRequest: req,
			AppConfig: config.NewMock(),
		}

		result, err := gridReq.GetSessionIDFromPath()
		assert.Error(t, err)
		assert.Empty(t, result)
	})
}

func TestRequest_HubPath(t *testing.T) {
	t.Run("when session is nil, returns an error", func(t *testing.T) {
		req := Request{}
		_, err := req.HubPath()
		assert.Error(t, err)
	})

	t.Run("when raw request is nil, returns an error", func(t *testing.T) {
		req := Request{
			Session: &Session{
				InternalID: &intSessionID,
				ExternalID: &extSessionID,
			},
		}
		_, err := req.HubPath()
		assert.Error(t, err)
	})

	t.Run("when everything is present, returns the url with the internal id", func(t *testing.T) {
		rawReq, err := http.NewRequest(http.MethodGet, existingUrl, nil)
		require.NoError(t, err)

		req := Request{
			Session: &Session{
				InternalID: &intSessionID,
				ExternalID: &extSessionID,
			},
			RawRequest: rawReq,
		}
		result, err := req.HubPath()
		require.NoError(t, err)

		expectedResult := fmt.Sprintf("%s/%s", sessionRoute, intSessionID)
		assert.NotNil(t, result)
		assert.Equal(t, expectedResult, result, "should have the internal id in the route")
	})
}

func TestRequest_HubHost(t *testing.T) {
	t.Run("when missing fields, returns error", func(t *testing.T) {
		t.Run("session", func(t *testing.T) {
			req := Request{}
			_, err := req.HubHost()
			assert.Error(t, err)
		})

		t.Run("hub", func(t *testing.T) {
			req := Request{
				Session: &Session{},
			}
			_, err := req.HubHost()
			assert.Error(t, err)
		})

		t.Run("port", func(t *testing.T) {
			req := Request{
				Session: &Session{
					Hub: &hub_registry.Hub{},
				},
			}
			_, err := req.HubHost()
			assert.Error(t, err)
		})

		t.Run("hostname", func(t *testing.T) {
			req := Request{
				Session: &Session{
					Hub: &hub_registry.Hub{
						Port: "1234",
					},
				},
			}
			_, err := req.HubHost()
			assert.Error(t, err)
		})
	})

	t.Run("when required fields are available, returns the correct host", func(t *testing.T) {
		t.Run("hostname", func(t *testing.T) {
			hostName := "http://something"
			hostPort := "5678"

			rawReq, err := http.NewRequest(http.MethodGet, existingUrl, nil)
			require.NoError(t, err)

			req := Request{
				Session: &Session{
					InternalID: &intSessionID,
					ExternalID: &extSessionID,
					Hub: &hub_registry.Hub{
						Hostname: hostName,
						Port: hostPort,
					},
				},
				RawRequest: rawReq,
			}

			result, err := req.HubHost()
			require.NoError(t, err)

			assert.Equal(t, fmt.Sprintf("%s:%s", hostName, hostPort), result)
		})

		t.Run("ip", func(t *testing.T) {
			hostName := "1.2.3.4"
			hostPort := "5678"

			rawReq, err := http.NewRequest(http.MethodGet, existingUrl, nil)
			require.NoError(t, err)

			req := Request{
				Session: &Session{
					InternalID: &intSessionID,
					ExternalID: &extSessionID,
					Hub: &hub_registry.Hub{
						IP: hostName,
						Port: hostPort,
					},
				},
				RawRequest: rawReq,
			}

			result, err := req.HubHost()
			require.NoError(t, err)

			assert.Equal(t, fmt.Sprintf("%s:%s", hostName, hostPort), result)
		})

	})
}

func TestRequest_GetInternalID(t *testing.T) {
	t.Run("valid path", func (t *testing.T) {
		rawReq, err := http.NewRequest(http.MethodGet, existingUrl, nil)
		require.NoError(t, err)

		req := Request{
			RawRequest: rawReq,
			AppConfig: config.NewMock(),
		}

		result, err := req.GetInternalID()
		assert.NoError(t, err)
		assert.Equal(t, intSessionID, result)
	})

	t.Run("invalid path", func (t *testing.T) {
		rawReq, err := http.NewRequest(http.MethodGet, "http://test.com/not/valid", nil)
		require.NoError(t, err)

		req := Request{
			RawRequest: rawReq,
			AppConfig: config.NewMock(),
		}

		_, err = req.GetInternalID()
		assert.Error(t, err)
	})
}

func TestRequest_GetExternalID(t *testing.T) {
	t.Run("valid path", func (t *testing.T) {
		rawReq, err := http.NewRequest(http.MethodGet, existingUrl, nil)
		require.NoError(t, err)

		req := Request{
			RawRequest: rawReq,
			AppConfig: config.NewMock(),
		}

		result, err := req.GetExternalID()
		assert.NoError(t, err)
		assert.Equal(t, extSessionID, result)
	})

	t.Run("invalid path", func (t *testing.T) {
		rawReq, err := http.NewRequest(http.MethodGet, "http://test.com/not/valid", nil)
		require.NoError(t, err)

		req := Request{
			RawRequest: rawReq,
			AppConfig: config.NewMock(),
		}

		_, err = req.GetExternalID()
		assert.Error(t, err)
	})
}

func TestRequest_GetHubID(t *testing.T) {
	t.Run("valid path", func (t *testing.T) {
		rawReq, err := http.NewRequest(http.MethodGet, existingUrl, nil)
		require.NoError(t, err)

		req := Request{
			RawRequest: rawReq,
			AppConfig: config.NewMock(),
		}

		result, err := req.GetHubID()
		assert.NoError(t, err)
		assert.Equal(t, hubID, result)
	})

	t.Run("invalid path", func (t *testing.T) {
		rawReq, err := http.NewRequest(http.MethodGet, "http://test.com/not/valid", nil)
		require.NoError(t, err)

		req := Request{
			RawRequest: rawReq,
			AppConfig: config.NewMock(),
		}

		_, err = req.GetHubID()
		assert.Error(t, err)
	})
}

func newTestRedis() *redismock.ClientMock {
	mr, err := miniredis.Run()
	if err != nil {
		panic(err)
	}

	client := redis.NewClient(&redis.Options{
		Addr: mr.Addr(),
	})

	return redismock.NewNiceMock(client)
}
