// Package session represents a Selenium Session.
package session

import (
	"bytes"
	"code.justin.tv/qe/grid_router/src/pkg/config"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	"encoding/json"
	"errors"
	"io/ioutil"
	"net/http"
	"strconv"
)

// Represents a Selenium Session
// ID is the assigned session id from a hub
// response is the New Response body passed back from Hub after POST /wd/session
type Session struct {
	ExternalID *string // External ID to Selenium Hub
	InternalID *string // Internal ID to Grid Router
	Hub        *hub_registry.Hub
	AppConfig  *config.Config
}

/*
Data Structs that pertain to new session responses
There are varying places data goes based on if the driver complies with w3c webdriver spec
For example, Chromedriver places sessionid at the root of a response, while geckodriver places it within "value"
Spec response data can be found here:
https://w3c.github.io/webdriver/#new-session
https://w3c.github.io/webdriver/#dfn-send-a-response
 */

// Structure to maintain the response from Hub.
// This would be the "value" key within the response. https://w3c.github.io/webdriver/#dfn-send-a-response
type newSessionResponseData struct {
	SessionID string
}

// Represents a New Session Response. Used for unmarshalling the JSON BOdy
type newSessionResponse struct {
	SessionID string // Non-W3C Spec Adhering Clients (ChromeDriver) place sid here
	Value     newSessionResponseData // Per W3C Spec, true response data is contained within value.
}

/**
Non-Structure Code Below
 */

// Populates a Session object based on an HTTP Response from the Hub
func (ses *Session) populateFromResponse(responseBody []byte) error {
	sid, err := ses.parseSessionID(responseBody) // Grab the SID from the body, ensure no error
	if err != nil {
		return err
	}

	extSid, err := ses.createExternalSID(sid) // Encode it and store it in the object
	if err != nil {
		return err
	}

	ses.ExternalID = &extSid
	ses.InternalID = &sid

	return nil
}

// Initializes a Session Struct with data based on the response from a Hub
// This is used when a session is created from the Hub, and its details are returned back as a response.
func (ses *Session) InitSessionFromResponse(resp *http.Response) error {
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	err = resp.Body.Close() // We no longer need that body, we'll replace it
	if err != nil {
		return err
	}

	resp.Body = ioutil.NopCloser(bytes.NewReader(body)) // Put the body back
	resp.ContentLength = int64(len(body))
	resp.Header.Set("Content-Length", strconv.Itoa(len(body)))

	err = ses.populateFromResponse(body)
	return err
}

// Parses the Session ID from a New Session Request
// body should be the body returned from POST /wd/session
// returns a string of the Session ID
func (ses *Session) parseSessionID(body []byte) (string, error) {
	var sessionID string

	sessionResponse := newSessionResponse{} // Create object to store the json results

	// Unmarshal the json body into the sesionResponse object
	err := json.Unmarshal(body, &sessionResponse)
	if err != nil {
		return "", err
	}

	// Based on W3C Spec Compliance, SessionID may be in two different places
	// { "sessionID":"mySID" } or { "value": { "sessionID":"mySID" }}
	// Chromedriver does not adhere to W3C Spec, thus we need to check both spots
	if sessionResponse.Value.SessionID != "" { // If not blank, assign value to sessionID var
		sessionID = sessionResponse.Value.SessionID
	} else if sessionResponse.SessionID != "" { // If not blank, assign value to sessionID var
		sessionID = sessionResponse.SessionID
	} else {
		ses.AppConfig.Logger.Error("The Session ID was found to be blank when parsing")
		ses.AppConfig.Logger.Debugf("Body that could not be parsed: %s", body)
		return "", errors.New("error! The Session ID was found to be blank")
	}

	return sessionID, nil
}

// Returns a Session ID that includes the Hub ID
func (ses *Session) createExternalSID(internalSid string) (string, error) {
	if ses.Hub == nil {
		return "", errors.New("the hub was nil")
	}
	if ses.Hub.ID == "" {
		return "", errors.New("the hub id was blank")
	}

	// Append the Hub ID at the end of the SID
	eSid := internalSid + "_" + ses.Hub.ID
	ses.AppConfig.Logger.Debugf("Encoding Sesion ID. Internal: [%s] External: [%s]", internalSid, eSid)
	return eSid, nil
}
