package main

import (
	"crypto/tls"
	"fmt"
	"github.com/go-redis/redis"
	"github.com/sirupsen/logrus"
	"os"
)

const HostENV = "REDIS_HOST"
const PortENV = "REDIS_PORT"
const DefaultHost = "localhost"
const DefaultPort = "6379"

// A DB Connection to Redis
type Redis struct {
	Client *redis.Client
	Host string
	Port string
	Logger *logrus.Logger
}

// Initializes a Redis Struct with values from environment variables
// Provide secrets so that the redis password can be used
func (c *Redis) Init(secrets Secrets, logger *logrus.Logger) {
	c.Logger = logger
	c.Host = c.getHost()
	c.Port = c.getPort()
	c.Client = c.createClient(secrets)
}

// Returns the Host of the Redis DB, based on the env variable REDIS_HOST. If not provided, uses a default
func (c *Redis) getHost() string {
	host := os.Getenv(HostENV)
	if len(host) == 0 {
		c.Logger.Warn("Env Var REDIS_HOST was not provided. Defaulting to localhost")
		host = DefaultHost
	}
	return host
}

// Returns the Port of the Redis DB, based on the env variable REDIS_PORT. If not provided, uses a default
func (c *Redis) getPort() string {
	port := os.Getenv(PortENV)
	if len(port) == 0 {
		c.Logger.Warn("Env Var REDIS_PORT was not provided. Defaulting to 6379")
		port = DefaultPort
	}
	return port
}

// Returns the URL of the Redis Instance, in the format of host:port
func (c *Redis) getURL() string {
	return fmt.Sprintf("%s:%s", c.Host, c.Port)
}

// Returns if the host of the Redis DB is localhost
func (c *Redis) isLocalhost() bool {
	return c.Host == DefaultHost
}

// Creates options for the Redis client, such as TLS/Authentication if not running on localhost
// Provide secrets so that the redis password can be used
func (c *Redis) createOptions(secrets Secrets) *redis.Options {
	redisOpts := &redis.Options{
		Addr: c.getURL(),
	}

	// If NOT Localhost, use Authentication and TLS
	if !c.isLocalhost() {
		redisOpts.Password = secrets.Redis
		redisOpts.TLSConfig = &tls.Config{}
	}
	return redisOpts
}

// Creates a redis client
// Provide secrets so that the redis password can be used
func (c *Redis) createClient(secrets Secrets) *redis.Client {
	redisOpts := c.createOptions(secrets)
	redisClient := redis.NewClient(redisOpts)

	// Ensure the redis client has a connection
	res, err := redisClient.Ping().Result()
	if err != nil || res != "PONG" {
		c.Logger.Warnf("Unable to establish initial connection with Redis database. Response: %v | Error: %v", res, err)
	} else {
		c.Logger.Info("Connection Established to Redis Server")
	}
	return redisClient
}
