package main

import (
	"fmt"
	"github.com/sirupsen/logrus/hooks/test"
	"github.com/stretchr/testify/assert"
	"os"
	"testing"
)

func TestGetHost(t *testing.T) {
	logger, _ := test.NewNullLogger()
	redis := Redis{
		Logger: logger,
	}
	// When not provided, defaults to localhost
	assert.Equal(t, DefaultHost, redis.getHost())

	// When provided, uses that
	expectedHost := "test123"
	err := os.Setenv(HostENV, expectedHost)
	assert.NoError(t, err)
	assert.Equal(t, expectedHost, redis.getHost())
}

func TestGetPort(t *testing.T) {
	logger, _ := test.NewNullLogger()
	redis := Redis{
		Logger: logger,
	}
	// When not provided, defaults to localhost
	assert.Equal(t, DefaultPort, redis.getPort())

	// When provided, uses that
	expectedPort := "1234"
	err := os.Setenv(PortENV, expectedPort)
	assert.NoError(t, err)
	assert.Equal(t, expectedPort, redis.getPort())
}

func TestIsLocalhost(t *testing.T) {
	logger, _ := test.NewNullLogger()
	redis := Redis{
		Logger: logger,
	}

	hostTest := []struct {
		in string
		out bool
	}{
		{"localhost", true},
		{"dev.amazonaws.com", false},
	}

	for _, tt := range hostTest {
		redis.Host = tt.in
		assert.Equal(t, tt.out, redis.isLocalhost())
	}
}

func TestCreateOptions(t *testing.T) {
	logger, _ := test.NewNullLogger()
	mockRedisPassword := "mockPassword"
	mockSecret := Secrets{}
	mockSecret.Redis = mockRedisPassword

	// When Localhost, addr should match, no password, and no tls config
	redis := Redis{
		Host: "localhost",
		Port: "1234",
		Logger: logger,
	}
	res := redis.createOptions(mockSecret)
	assert.Equal(t, fmt.Sprintf("%s:%s", redis.Host, redis.Port), res.Addr) // Addr should be host:port
	assert.Equal(t, "", res.Password)
	assert.Nil(t, res.TLSConfig)

	// When not localhost, addr should match, has a password, and has tls config
	redis = Redis{
		Host: "dev.amazonaws.com",
		Port: "1234",
		Logger: logger,
	}
	res = redis.createOptions(mockSecret)
	assert.Equal(t, fmt.Sprintf("%s:%s", redis.Host, redis.Port), res.Addr) // Addr should be host:port
	assert.Equal(t, mockRedisPassword, res.Password)
	assert.NotNil(t, res.TLSConfig)
}
