package main

import (
	"code.justin.tv/hygienic/amzncorp"
	"code.justin.tv/systems/sandstorm/callback"
	"code.justin.tv/systems/sandstorm/manager"
	"errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/sirupsen/logrus"
	"strings"
	"time"
)

// Create some variables for the secret names
var authTokenSecretName = "qa-eng/grid-router/production/api_access_key"
var redisSecretName = "qa-eng/grid-router/production/redis"
var roleARN = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/grid_router"

// Application Secrets fetched from the secret manager
type Secrets struct {
	APIKeys []string
	Redis   string
	Manager SecretManager
	Logger  *logrus.Logger
	config  *aws.Config
}

// Interface that allows for working with the sandstorm manager
type SecretManager interface {
	Get(secretName string) (*manager.Secret, error)
	RegisterSecretUpdateCallback(secretName string, callback func(input manager.SecretChangeset))
	ListenForUpdates() (err error)
}

// Initializes a Secrets struct to fill in the values
func (s *Secrets) Init(logger *logrus.Logger, localRun bool) error {
	// Initialize the manager if it was nil
	if s.Manager == nil {
		s.config = GetAWSConfig(localRun)
		s.Manager = GetSecretManager(s.config)
	}

	if s.Logger == nil {
		s.Logger = logger
	}

	// Fill the API Keys
	err := s.configureAPIKeys()
	if err != nil {
		return err
	}

	// Fill the Redis Key
	err = s.configureRedis()
	if err != nil {
		return err
	}

	err = s.Manager.ListenForUpdates()
	return err
}

// Returns an AWS Config with the proper credentials
func GetAWSConfig(localRun bool) *aws.Config {
	if localRun { // If running locally, use Isengard to authenticate
		creds := credentials.NewCredentials(&amzncorp.IsengardCredentials{
			AWSAccountID: "425992774280",
			IAMRoleName: "Admin",
		})
		return manager.AWSConfig(creds, roleARN) // provide those creds
	} else {
		return manager.AWSConfig(nil, roleARN) // no creds to provide
	}
}

// Returns a Secret Manager with the config provided
func GetSecretManager(config *aws.Config) *manager.Manager {
	return manager.New(manager.Config{
		AWSConfig: config,
	})
}

// Configures API Keys
func (s *Secrets) configureAPIKeys() error {
	s.watchForUpdate(authTokenSecretName, s.updateAPIKeys)
	return s.updateAPIKeys()
}

// Configures Redis Key
func (s *Secrets) configureRedis() error {
	s.watchForUpdate(redisSecretName, s.updateRedisKeys)
	return s.updateRedisKeys()
}

// Updates the API Keys in the Struct
func (s *Secrets) updateAPIKeys() error {
	keys, err := s.fetchAPIKeys()
	if err != nil {
		return err
	}

	s.APIKeys = keys
	return nil
}

// Grabs the API Keys from the Secret Manager and parses them into an array
func (s *Secrets) fetchAPIKeys() ([]string, error) {
	var parsedSecrets []string

	s.Logger.Info("Fetching API Keys from Sandstorm")
	secret, err := s.Manager.Get(authTokenSecretName)
	if err != nil {
		return parsedSecrets, err
	}
	if secret == nil {
		return parsedSecrets, errors.New("secret was nil")
	}

	secretContent := string(secret.Plaintext) // Grab the content as a string
	return strings.Split(secretContent, ","), nil // Store it as an array, split by ","
}

// Updates the Redis Keys within the struct
func (s *Secrets) updateRedisKeys() error {
	secret, err := s.Manager.Get(redisSecretName)
	if err != nil {
		return err
	}
	if secret == nil {
		return errors.New("secret was nil")
	}

	s.Redis = string(secret.Plaintext)
	return nil
}

// updateSecret takes a secret and updates it within the struct
type updateSecret func() error

// Sets up a secret to be watched for updates
// Pass the name of the secret, and a function that should be ran if the secret content changes
func (s *Secrets) watchForUpdate(secretName string, updateFN updateSecret) {
	// Set up a callback for if the secret updates, and listen for those updates. Checks every 10 seconds
	dw := &callback.Dweller{
		Duration: 10 * time.Second,
	}

	// Create a callback
	secretCb := func(input manager.SecretChangeset) {
		if input.SecretsHaveChanged() {
			if input.SecretHasBeenUpdated(secretName) {
				err := updateFN()
				if err != nil {
					s.Logger.Warnf("Problem updating Secret %s. Received err: %v", secretName, err)
				}
			}
		}
	}

	// Register the callback
	dwellCb := dw.Dwell(secretCb)
	s.Manager.RegisterSecretUpdateCallback(secretName, dwellCb)
}
