package main

import (
	"code.justin.tv/qe/grid_router/src/mocks"
	sandstorm "code.justin.tv/systems/sandstorm/manager"
	"errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/sirupsen/logrus/hooks/test"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"strings"
	"testing"
)

func TestSecrets_Init(t *testing.T) {
	expectedKeys := []string{"mockAPI1", "mockAPI2"}
	expectedRedisKey := "redistest"

	// Create a Mock Secret Manager
	mockSecManager := &mocks.SecretManager{}
	manager := Secrets{
		Manager: mockSecManager,
	}

	// Create a mock logger
	logger, _ := test.NewNullLogger()

	// Create the Mock Secrets
	mockAPIKeySecret := &sandstorm.Secret{
		Plaintext: []byte(strings.Join(expectedKeys, ",")),
	}
	mockRedisSecret := &sandstorm.Secret{
		Plaintext: []byte(expectedRedisKey),
	}

	// Mock return secrets for auth tokens
	mockSecManager.On("Get", authTokenSecretName).Return(mockAPIKeySecret, nil).Once()
	mockSecManager.On("RegisterSecretUpdateCallback", authTokenSecretName, mock.Anything).Return()

	// Mock return secrets for redis
	mockSecManager.On("Get", redisSecretName).Return(mockRedisSecret, nil).Once()
	mockSecManager.On("RegisterSecretUpdateCallback", redisSecretName, mock.Anything).Return()

	// Mock listening for updates
	mockSecManager.On("ListenForUpdates").Return(nil).Once()

	// Initialize and expect the secrets to be filled
	err := manager.Init(logger, false)
	assert.NoError(t, err)
	assert.Equal(t, manager.APIKeys, expectedKeys)
	assert.Equal(t, manager.Redis, expectedRedisKey)

	/*
	When sandstorm returns an error, it should be returned
	 */
	 // ListenForUpdates returns error
	mockSecManager.ExpectedCalls = mockSecManager.ExpectedCalls[:0] // Clear the expected calls
	mockSecManager.On("RegisterSecretUpdateCallback", authTokenSecretName, mock.Anything).Return().Once()
	mockSecManager.On("RegisterSecretUpdateCallback", redisSecretName, mock.Anything).Return().Once()

	mockSecManager.On("Get", authTokenSecretName).Return(mockAPIKeySecret, nil).Once()
	mockSecManager.On("Get", redisSecretName).Return(mockRedisSecret, nil).Once()
	mockSecManager.On("ListenForUpdates").Return(errors.New("testlistenerr")).Once()
	err = manager.Init(logger, false)
	assert.EqualError(t, err, "testlistenerr")

	// API Secrets Return Error
	mockSecManager.ExpectedCalls = mockSecManager.ExpectedCalls[:0] // Clear the expected calls
	mockSecManager.On("RegisterSecretUpdateCallback", authTokenSecretName, mock.Anything).Return().Once()
	mockSecManager.On("RegisterSecretUpdateCallback", redisSecretName, mock.Anything).Return().Once()

	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	mockSecManager.On("Get", authTokenSecretName).Return(mockAPIKeySecret, errors.New("testGetErr")).Once()
	mockSecManager.On("Get", redisSecretName).Return(mockRedisSecret, nil).Once()
	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	err = manager.Init(logger, false)
	assert.EqualError(t, err, "testGetErr")

	// Redis Secret Return Error
	mockSecManager.ExpectedCalls = mockSecManager.ExpectedCalls[:0] // Clear the expected calls
	mockSecManager.On("RegisterSecretUpdateCallback", authTokenSecretName, mock.Anything).Return().Once()
	mockSecManager.On("RegisterSecretUpdateCallback", redisSecretName, mock.Anything).Return().Once()

	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	mockSecManager.On("Get", authTokenSecretName).Return(mockAPIKeySecret, nil).Once()
	mockSecManager.On("Get", redisSecretName).Return(mockRedisSecret, errors.New("testGetErr")).Once()
	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	err = manager.Init(logger, false)
	assert.EqualError(t, err, "testGetErr")

	// API Secrets Return Nil
	mockSecManager.ExpectedCalls = mockSecManager.ExpectedCalls[:0] // Clear the expected calls
	mockSecManager.On("RegisterSecretUpdateCallback", authTokenSecretName, mock.Anything).Return().Once()
	mockSecManager.On("RegisterSecretUpdateCallback", redisSecretName, mock.Anything).Return().Once()

	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	mockSecManager.On("Get", authTokenSecretName).Return(nil, nil).Once()
	mockSecManager.On("Get", redisSecretName).Return(mockRedisSecret, nil).Once()
	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	err = manager.Init(logger, false)
	assert.EqualError(t, err, "secret was nil")

	// Redis Secret Returns Nil
	mockSecManager.ExpectedCalls = mockSecManager.ExpectedCalls[:0] // Clear the expected calls
	mockSecManager.On("RegisterSecretUpdateCallback", authTokenSecretName, mock.Anything).Return().Once()
	mockSecManager.On("RegisterSecretUpdateCallback", redisSecretName, mock.Anything).Return().Once()

	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	mockSecManager.On("Get", authTokenSecretName).Return(mockAPIKeySecret, nil).Once()
	mockSecManager.On("Get", redisSecretName).Return(nil, nil).Once()
	mockSecManager.On("ListenForUpdates").Return(nil).Once()
	err = manager.Init(logger, false)
	assert.EqualError(t, err, "secret was nil")
}

func TestGetAWSConfig(t *testing.T) {
	t.Run("localRun true", func (t *testing.T) {
		result := GetAWSConfig(true)
		require.NotNil(t, result)
		assert.NotNil(t, result.Credentials)
	})

	t.Run("localRun false", func (t *testing.T) {
		result := GetAWSConfig(false)
		require.NotNil(t, result)
		assert.NotNil(t, result.Credentials)
	})
}

func TestGetSecretManager(t *testing.T) {
	config := &aws.Config{}
	t.Run("returns a manager with the proper config", func (t *testing.T) {
		result := GetSecretManager(config)
		require.NotNil(t, result)
		require.NotNil(t, result.Config)
		assert.Equal(t, config, result.Config.AWSConfig)
	})
}
