package amzncorp

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"path/filepath"
	"runtime"
	"sync"
	"time"

	"github.com/go-ini/ini"

	"github.com/aws/aws-sdk-go/aws/credentials"
)

// IsengardCredentials allows authentication via mwinit and isengard
type IsengardCredentials struct {
	// Create this via MidwayClient.  It is optional.  If you do not set it, it assumes a cookie location at ~/.midway/cookie
	MidwayClient *http.Client

	// AWSAccountID the numeric ID of your account
	AWSAccountID string
	// Role to assume
	IAMRoleName string
	// Profile is the name of the profile inside your ~/.aws/config or ~/.aws/credentials file
	// It will populate AWSAccountID with either "account" or "account_id" and IAMRoleName with either "role" or "role_name"
	// This logic is similar to SharedCredentialsProvider
	Profile string

	// This is optional and logs any internal errors that cannot be simply returned
	OnErr func(error)

	once       sync.Once
	initErrors error
	credentials.Expiry
}

var _ credentials.Provider = (*IsengardCredentials)(nil)

// Retrieve satisfies the credentials.Provider interface by getting credentials from isengard
func (c *IsengardCredentials) Retrieve() (credentials.Value, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	return c.retrieveContext(ctx)
}

type isengardGetAssumeRoleCredentialsRequest struct {
	AWSAccountID string `json:"AWSAccountID"`
	IAMRoleName  string `json:"IAMRoleName"`
}

type isengardGetAssumeRoleCredentialsResponse struct {
	AssumeRoleResult string `json:"AssumeRoleResult"`
}

type isengardAssumeRoleResult struct {
	SDKResponseMetadata isengardSDKResponseMetadata `json:"sdkResponseMetadata"`
	SDKHttpMetadata     isengardSDKHttpMetadata     `json:"sdkHttpMetadata"`
	Credentials         assumeRoleCredentials       `json:"credentials"`
	AssumedRoleUser     assumeRoleUser              `json:"assumedRoleUser"`
}

type isengardSDKResponseMetadata struct {
	RequestID string `json:"requestId"`
}

type isengardSDKHttpMetadata struct {
	HTTPHeaders    map[string]string `json:"httpHeaders"`
	HTTPStatusCode int               `json:"httpStatusCode"`
}

type assumeRoleCredentials struct {
	AccessKeyID      string `json:"accessKeyId"`
	SecretAccessKey  string `json:"secretAccessKey"`
	SessionToken     string `json:"sessionToken"`
	ExpirationMillis int64  `json:"expiration"`
}

type assumeRoleUser struct {
	AssumedRoleID string `json:"assumedRoleId"`
	ARN           string `json:"arn"`
}

func call(ctx context.Context, client *http.Client, url string, target string, input, output interface{}, onErr func(error)) error {
	buf, err := json.Marshal(input)
	if err != nil {
		return err
	}

	req, err := http.NewRequest("POST", url, bytes.NewReader(buf))
	if err != nil {
		return err
	}
	req.Header.Set("X-Amz-Target", target)
	req.Header.Set("Content-Encoding", "amz-1.0")
	req.Header.Set("Content-Type", "application/json; charset=UTF-8")
	req = req.WithContext(ctx)

	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer func() {
		if closeBodyErr := resp.Body.Close(); closeBodyErr != nil && onErr != nil {
			onErr(closeBodyErr)
		}
	}()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	return json.Unmarshal(body, output)
}

func (c *IsengardCredentials) retrieveContext(ctx context.Context) (credentials.Value, error) {
	var v credentials.Value
	c.once.Do(func() {
		if c.MidwayClient == nil {
			asFile, err := os.Open(path.Join(os.Getenv("HOME"), ".midway/cookie"))
			if err != nil {
				c.initErrors = err
				return
			}
			c.MidwayClient, c.initErrors = MidwayClient(asFile)
			if err := asFile.Close(); err != nil && c.OnErr != nil {
				c.OnErr(err)
			}
		}
	})
	if c.initErrors != nil {
		return v, c.initErrors
	}

	input, createCredsErr := c.createCredentials()
	if createCredsErr != nil {
		return v, createCredsErr
	}
	output := new(isengardGetAssumeRoleCredentialsResponse)

	err := call(ctx, c.MidwayClient,
		"https://isengard-service.amazon.com",
		"IsengardService.GetAssumeRoleCredentials",
		input, output, c.OnErr)
	if err != nil {
		return v, err
	}

	result := new(isengardAssumeRoleResult)

	err = json.Unmarshal([]byte(output.AssumeRoleResult), result)
	if err != nil {
		return v, err
	}

	v.AccessKeyID = result.Credentials.AccessKeyID
	v.SecretAccessKey = result.Credentials.SecretAccessKey
	v.SessionToken = result.Credentials.SessionToken
	v.ProviderName = "IsengardProvider"

	expTime := time.Unix(result.Credentials.ExpirationMillis/int64(time.Second/time.Millisecond), 0).UTC()
	c.Expiry.SetExpiration(expTime, 10*time.Second)

	return v, nil
}

func (c *IsengardCredentials) createCredentials() (*isengardGetAssumeRoleCredentialsRequest, error) {
	ret := &isengardGetAssumeRoleCredentialsRequest{
		AWSAccountID: c.getAccountID(),
		IAMRoleName:  c.getRoleName(),
	}
	if ret.AWSAccountID == "" {
		return nil, errors.New("unable to discover account ID")
	}
	if ret.IAMRoleName == "" {
		return nil, errors.New("unable to discover role name")
	}
	return ret, nil
}

func (c *IsengardCredentials) getAccountID() string {
	if c.AWSAccountID != "" {
		return c.AWSAccountID
	}
	accountINIKeys := []string{
		"account",
		"account_id",
	}
	return c.iniFetch(accountINIKeys)
}

func (c *IsengardCredentials) getRoleName() string {
	if c.IAMRoleName != "" {
		return c.IAMRoleName
	}
	awsProfile := c.getProfile()
	if awsProfile == "" {
		return ""
	}
	roleINIKeys := []string{
		"role",
		"role_name",
	}
	return c.iniFetch(roleINIKeys)
}

func (c *IsengardCredentials) iniFetch(iniKeys []string) string {
	awsProfile := c.getProfile()
	if awsProfile == "" {
		return ""
	}
	for _, filename := range c.getCredentialFiles() {
		config, err := ini.Load(filename)
		if err != nil {
			continue
		}
		possibleSections := []string{
			awsProfile,
			"profile " + awsProfile,
		}
		for _, section := range possibleSections {
			iniProfile, err := config.GetSection(section)
			if err != nil {
				continue
			}
			for _, accountKey := range iniKeys {
				if getRes, err := iniProfile.GetKey(accountKey); err == nil && getRes.String() != "" {
					return getRes.String()
				}
			}
		}
	}
	return ""
}

func (c *IsengardCredentials) getProfile() string {
	ret := c.Profile
	if ret == "" {
		ret = os.Getenv("AWS_PROFILE")
	}
	if ret == "" {
		ret = "default"
	}

	return ret
}

func (c *IsengardCredentials) getCredentialFiles() []string {
	return []string{
		os.Getenv("AWS_SHARED_CREDENTIALS_FILE"),
		filepath.Join(userHomeDir(), ".aws", "credentials"),
		filepath.Join(userHomeDir(), ".aws", "config"),
	}
}

// UserHomeDir returns the home directory for the user the process is
// running under.
func userHomeDir() string {
	if runtime.GOOS == "windows" { // Windows
		return os.Getenv("USERPROFILE")
	}

	// *nix
	return os.Getenv("HOME")
}
