package naiive

import (
	"fmt"
	"regexp"
	"strings"

	"code.justin.tv/tshadwell/nice"
)

var testMap = make(map[string]Test)

func TestShortNames() (shortNames []string) {
	for _, test := range DefaultTests {
		shortNames = append(shortNames, test.ShortName)
	}

	return
}

type ErrNoSuchTest string

func (e ErrNoSuchTest) Error() string { return fmt.Sprintf("no such test %+q", string(e)) }

func TestByShortName(shortName string) (t Test, err error) {
	var ok bool
	if t, ok = testMap[shortName]; !ok {
		err = ErrNoSuchTest(shortName)
	}

	return
}

func registerTest(t Test) Test {
	DefaultTests = append(DefaultTests, t)

	testMap[t.ShortName] = t
	return t
}

var TestUnlimitedRead = registerTest(Test{
	Name:       "Unlimited Read From External Input",
	ShortName:  "UnlimitedRead",
	Regexp:     regexp.MustCompile(`(?:Read(?:All)?|(?:New)?Decoder?)\(\w+\.Body\)`),
	Confidence: nice.LowConfidence,
	Severity:   nice.MediumSeverity,

	Desc: `Go's encoding/* parsers don't limit how much input they can take, and, as such they
will read from their inputs forever until the software runs out of memory and kills itself.

Go's input pipelines, such as "net/http".Request.Body do not limit how much input they can take by
default, and as such bad clients can push information to them forever.

These issues combined can cause an effective denial of service attack on Go systems.`,

	Examples: []string{
		"Read(w.Body)",
		"ReadAll(w.Body)",
		"NewDecoder(w.Body)",
		"Decode(w.Body)",
	},
})

var TestBulldozingError = registerTest(Test{
	Name:       "Bulldozing Error",
	ShortName:  "BulldozingError",
	Regexp:     regexp.MustCompile(`if.*err\s*!=\s*nil\s*{(?:[^r}]|r(?:[^e}]|e(?:[^t}]|t(?:[^u}]|u(?:[^r}]|r(?:[^n}]))))))+}`),
	Confidence: nice.LowConfidence,
	Severity:   nice.HighSeverity,

	Desc: `Because Go requires explicit handling of all errors, it's common to see issues where an error is checked, and
the appropriate error handler is called, but the execution flow is not halted because a return is missing.

This can mean that, for example, an unauthenticated user can perform an authenticated action despite seeing a failure message.`,

	Examples: []string{
		`if err != nil {
	w.TrustedError(err)
}`},
	NonExamples: []string{
		`if err != nil {
	w.TrustedError(err)
	return
}`},
})

var TestUnsafeTemplateUsage = registerTest(Test{
	Name:       "Unsafe Template Usage",
	Regexp:     regexp.MustCompile(`template\.(?:HTML|HTMLAttr|JS|JSStr)\b`),
	ShortName:  "UnsafeTemplate",
	Confidence: nice.MediumConfidence,
	Severity:   nice.HighSeverity,

	Desc: `html/template exposes a number of special types that are used to bypass certian aspects of
the context aware escaping that protects against XSS.

By-and-large usage of these types is unnecessary, and can usually be avoided by, rather than
passing HTML to the templating system, creating a template for the HTML fragment that would be passed to the
templating system. For example rather than using:

	<div class="userProfile">
		{{.UserProfileHTML}}
	</div>

You can instead define a template that represents the HTML in the user profile, and reference it, using
two separate templates:

	template.Must(template.New("user-profile").Parse(
		<h1>{{user.Name}}</h1>
		<div class="bio">{{.user.Bio}}</div>
	))


	template.Must(template.New("profile-page").Parse(
		<div class="userProfile">
			{{template "user-profile" .user}}
		</div>
	))

In this contrived example, these templates could equally be merged into one, describing the whole HTML tree
but these bugs usually result from HTML that would be too unweildy if expressed as a single template.`,

	Examples:    []string{"template.HTML", "template.HTMLAttr", "template.JS", "template.JSStr"},
	NonExamples: []string{"template.New"},
})

var TestHardCodedKey = registerTest(Test{
	Name: "Hard Coded Key",
	// stolen from truffle hog
	Regexp: regexp.MustCompile("(?:" + strings.Join([]string{
		`(?:xox[p|b|o|a]-[0-9]{12}-[0-9]{12}-[0-9]{12}-[a-z0-9]{32})`, // slack token
		`-----BEGIN RSA PRIVATE KEY-----`,                             // ssh rsa private key
		`-----BEGIN OPENSSH PRIVATE KEY-----`,                         // ssh openssh private key
		`-----BEGIN DSA PRIVATE KEY-----`,                             // you get the idea
		`-----BEGIN EC PRIVATE KEY-----`,
		`-----BEGIN PGP PRIVATE KEY BLOCK-----`,                               // private key block
		`[f|F][a|A][c|C][e|E][b|B][o|O][o|O][k|K].*['|\"][0-9a-f]{32}['|\"]`,  // facebook oauth
		`[t|T][w|W][i|I][t|T][t|T][e|E][r|R].*['|\"][0-9a-zA-Z]{35,44}['|\"]`, // twitter oauth
		`[g|G][i|I][t|T][h|H][u|U][b|B].*['|\"][0-9a-zA-Z]{35,40}['|\"]`,      // github oauth
		`(?:"client_secret":"[a-zA-Z0-9-_]{24}")`,                             // google oauth (i dunno why it has a non capturing group)
		`AKIA[0-9A-Z]{16}`, // AWS API key
		`[h|H][e|E][r|R][o|O][k|K][u|U].*[0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12}`, // heroku api key
		//		`s|S][e|E][c|C][r|R][e|E][t|T].*['|\"][0-9a-zA-Z]{32,45}['|\"]`, // generic secret [looks noisy so disabled]
		//		`[a|A][p|P][i|I][_]?[k|K][e|E][y|Y].*['|\"][0-9a-zA-Z]{32,45}['|\"]`, // generic api key [looks noisy so disabled]
		`https://hooks.slack.com/services/T[a-zA-Z0-9_]{8}/B[a-zA-Z0-9_]{8}/[a-zA-Z0-9_]{24}`, // slack webhook
		`"type": "service_account"`, // google service account
		`SK[a-z0-9]{32}`,            // twilio api key
		`[a-zA-Z]{3,10}://[^/\\s:@]{3,20}:[^/\\s:@]{3,20}@.{1,100}[\"'\\s]`, // password in URL
	}, "|") + ")"),
	ShortName:  "Credentials",
	Confidence: nice.MediumConfidence,
	Severity:   nice.MediumSeverity,

	Desc: `Credentials that are written directly into the code allow attackers with access to the code direct access to the systems it controls. Additionally, upon compromise of thise credentials, manual intervention is required to rotate all instances of these secrets. Instead use a secret management system (sandstorm).`,

	Examples:    []string{"-----BEGIN RSA PRIVATE KEY-----"},
	NonExamples: []string{},
})

var TestInsecureCryptography = registerTest(Test{
	Name:       "Insecure cryptography",
	ShortName:  "InsecureCryptography",
	Regexp:     regexp.MustCompile(`crypto/(?:md5|rc4|des)`),
	Confidence: nice.LowConfidence,
	Severity:   nice.MediumSeverity,

	Desc: `MD5, RC4 and DES are almost always a bad use cryptography and aren't useful anymore. Check out our crypto traning in the Appsec Video Library: https://wiki.twitch.com/display/SEC/Appsec+Video+Library`,

	Examples: []string{"crypto/md5", "crypto/des", "crypto/rc4"},
})

var TestSQLConcatenation = registerTest(Test{
	Name:       "SQL String Concatenation",
	ShortName:  "SQLConcat",
	Regexp:     regexp.MustCompile(`(?:SELECT|DELETE|INSERT|UPDATE|INTO|FROM|WHERE)[^"]+"\s*\+`),
	Confidence: nice.HighConfidence,
	Severity:   nice.HighSeverity,

	Desc: `SQL injection, allowing an attacker full control over a database is endemic where SQL strings are built manually. Use prepared statements, or let your DBMS driver generate the SQL strings.`,

	Examples: []string{`SELECT FROM X WHERE " + val`},
})

var DefaultTests = Tests{}
