//Package naiive implements a simple regex based static analysis tool for Go
package naiive // import "code.justin.tv/tshadwell/nice/naiive"

import (
	"fmt"
	"go/token"
	"io/ioutil"
	"os"
	"path/filepath"
	"regexp"
	"runtime"
	"strings"
	"sync"

	"code.justin.tv/tshadwell/nice"
	"code.justin.tv/tshadwell/nice/progress"
	"github.com/golang/glog"
)

type Test struct {
	*regexp.Regexp
	Name       string
	Confidence nice.Confidence
	Severity   nice.Severity

	Desc        string
	Examples    []string
	NonExamples []string
	ShortName   string
}

var _ nice.Tester = Tests{}

func (t Test) TestName() string                { return t.Name }
func (t Test) TestConfidence() nice.Confidence { return t.Confidence }
func (t Test) TestDescription() string         { return fmt.Sprintf("regexp %s\n%s", t.Regexp, t.Desc) }
func (t Test) TestSeverity() nice.Severity     { return t.Severity }

type TestConfig struct {
	IncludeStdLib bool
	CombinedRegex *regexp.Regexp
	Tests

	*progress.Tracker
}

func CombineRegex(prefix string, r ...*regexp.Regexp) (combined *regexp.Regexp, err error) {
	var regexParts []string
	for _, re := range r {
		if sub := re.NumSubexp(); sub != 0 {
			err = fmt.Errorf("cannot combine regex %s, which has %d subexpression(s) and cannot have any.", re, sub)
			return
		}

		regexParts = append(regexParts, "("+re.String()+")")
	}

	return regexp.Compile(prefix + "(?:" + strings.Join(regexParts, "|") + ")")
}

func (t *TestConfig) combineRegex() (err error) {
	var re = make([]*regexp.Regexp, len(t.Tests))
	for i, v := range t.Tests {
		re[i] = v.Regexp
	}

	if t.CombinedRegex, err = CombineRegex("", re...); err != nil {
		return
	}

	return
}

func (t *TestConfig) NiceTest(p *nice.Program) (findings nice.Findings, err error) {
	if t.CombinedRegex == nil {
		if err = t.combineRegex(); err != nil {
			return
		}

		if glog.V(3) {
			glog.Infof("rendered combined regex %s", t.CombinedRegex)
		}
	}

	ld, err := p.Ast()
	if err != nil {
		return
	}

	buildContext, err := p.BuildContext()
	if err != nil {
		return
	}

	var errors nice.Errors
	var errorsLock sync.Mutex

	var findingLock sync.Mutex

	var wg sync.WaitGroup

	var recordError = func(err error) {
		errorsLock.Lock()
		errors = append(errors, err)
		errorsLock.Unlock()
	}

	var recordFinding = func(f nice.Findings) {
		findingLock.Lock()
		findings = append(findings, f...)
		findingLock.Unlock()
	}

	var files []*token.File
	ld.Fset.Iterate(func(f *token.File) bool { files = append(files, f); return true })

	glog.V(4).Infof("loaded %d files: %+v", len(files), files)

	scanTracker := t.Tracker.NewChild()
	scanTracker.Track(progress.ValueStatus{
		Max:  len(files),
		Text: "scan",
	})

	var fileDone = func() {
		scanTracker.Track(progress.ValueStatus{Delta: 1})
	}

	var CheckFile = func(f *token.File) (findings nice.Findings, err error) {
		defer fileDone()
		filename := f.Name()
		if filepath.Base(filename) == "C" ||
			// filepath says HasPrefix is bad but offers no better option.
			// considering the paths are all canonical here i reckon it wont be an issue
			// could use filepath.Match but it seems a bit overkill 🤔
			(!t.IncludeStdLib && strings.HasPrefix(filename, buildContext.GOROOT)) {
			return nil, nil
		}
		glog.V(3).Info("[naiive] checking file", filename)

		rwf, err := os.Open(filename)
		if err != nil {
			glog.V(3).Infof("failed to open file %s", filename)
			return
		}

		defer rwf.Close()

		bt, err := ioutil.ReadAll(rwf)
		if err != nil {
			return
		}

		matches := t.CombinedRegex.Copy().FindAllSubmatchIndex(bt, -1)
		for _, matchGroup := range matches {
			matchGroup = matchGroup[2:] // strip off 'whole regex match'
			// CombineRegex() created match groups corresponding to each regex test
			// here we iterate over the ith test result, interpreting the (rather complex)
			// output of FindAllSubmatchIndex.
			for i, ed := 0, len(t.Tests); i < ed; i++ {
				m := matchGroup[i*2 : i*2+2]
				if m[0] == -1 {
					continue
				}

				if sz := f.Size(); m[0] > sz || m[1] > sz {
					glog.V(3).Infof("somehow %s has a finding at %d/%d larger than the file (%d)", filename, m[0], m[1], sz)
					continue
				}

				finding := nice.Finding{
					Position: [2]token.Position{
						f.Position(f.Pos(m[0])),
						f.Position(f.Pos(m[1])),
					},

					Describer: t.Tests[i],
				}

				findings = append(findings, finding)
			}
		}

		return

	}

	var queue = make(chan *token.File)

	for i, nRoutines := 0, runtime.NumCPU(); i < nRoutines; i++ {
		wg.Add(1)
		go func(n int) {
			routineTracker := scanTracker.NewChild()

			routineTracker.Track(progress.ValueStatus{
				Max:  len(files) / nRoutines,
				Type: progress.IndefiniteValueType,
			})

			for file := range queue {
				routineTracker.Track(progress.ValueStatus{
					Text: p.MustCleanPath(file.Name()),
				})
				if findings, err := CheckFile(file); err != nil {
					recordError(err)
				} else {
					recordFinding(findings)
				}
			}
			wg.Done()
		}(i)
	}

	for _, file := range files {
		queue <- file
	}

	close(queue)

	wg.Wait()

	if len(errors) > 0 {
		err = errors
	}

	return
}

type Tests []Test

func (t Tests) ConfidenceRange(min nice.Confidence, max nice.Confidence) (filtered Tests) {
	for _, v := range t {
		if (min == -1 || v.Confidence >= min) && (max == -1 || v.Confidence <= max) {
			filtered = append(filtered, v)
		}
	}

	return
}

func (t Tests) SeverityRange(min nice.Severity, max nice.Severity) (filtered Tests) {
	for _, v := range t {
		if (min == -1 || v.Severity >= min) && (max == -1 || v.Severity <= max) {
			filtered = append(filtered, v)
		}
	}

	return
}

func (t Tests) NiceTest(p *nice.Program) (findings nice.Findings, err error) {
	return (&TestConfig{Tests: t}).NiceTest(p)
}
