package progress

import (
	"fmt"
	"time"
)

type StatusType uint

const (
	ValueType StatusType = iota
	IndefiniteValueType
	ChildType
)

type Status interface {
	StatusType() StatusType
}

type ChildStatus struct {
	New  *Tracker
	Sub  chan chan<- Status
	done chan<- *Tracker
}

func (ChildStatus) StatusType() StatusType { return ChildType }

type ValueStatus struct {
	Type       StatusType
	Value, Max int
	Delta      int
	Text       string
	Elapsed    time.Duration
}

func (v ValueStatus) StatusType() StatusType { return v.Type }

type Tracker struct {
	ChSubscribe chan chan<- Status
	Unsubscribe chan chan<- Status

	Subscribers map[chan<- Status]bool

	ChTrack chan Status

	Timeout time.Duration

	Close chan bool

	closeChildren []chan<- bool

	value, max int
	text       string

	last time.Time
}

func (t Tracker) String() string { return fmt.Sprintf("%p", t.ChTrack) }

func (t *Tracker) Subscribe() <-chan Status {
	var ch = make(chan Status)
	t.ChSubscribe <- ch

	return ch
}

// Track tracks this status. if this is a nil tracker, this is a no-op.
func (p *Tracker) Track(s Status) {
	if p == nil {
		return
	}

	if p.ChTrack == nil {
		panic("incorrectly initialised Tracker")
	}

	p.ChTrack <- s
}

func (p *Tracker) Go() {
	p.last = time.Now()
	if p.Subscribe == nil || p.Unsubscribe == nil {
		panic("incorrectly initialised Tracker")
	}
	for {
		select {
		case <-p.Close:
			close(p.Close)
			close(p.ChSubscribe)
			for ch, _ := range p.Subscribers {
				delete(p.Subscribers, ch)
				close(ch)
			}

			for _, ch := range p.closeChildren {
				go func() {
					ch <- true
				}()
			}
		case sub := <-p.ChSubscribe:
			p.Subscribers[sub] = true

		case status := <-p.ChTrack:
			if len(p.Subscribers) == 0 {

				panic(fmt.Sprintf("-> broadcast %+v to %d subs: %+v (%p)", status, len(p.Subscribers), p.Subscribers, p.ChTrack))
			}

			switch v := status.(type) {
			case ChildStatus:
				// this is kind of a mess.
				// we need the subscriptions for children to to thru
				// before any events are handled, which means
				// we need to prevent the new child for being returned
				// until the subs are sending. So when the listener
				// gets a ChildStatus, we use the done field
				// to tell the NewChild func when the subs are sent.
				//
				// the subs, equally close the subs field when
				// they've sent their subs.
				p.closeChildren = append(p.closeChildren, v.New.Close)
				for sub, _ := range p.Subscribers {
					subProxy := make(chan chan<- Status)
					sub <- ChildStatus{
						New: v.New,
						Sub: subProxy,
					}

					for newSub := range subProxy {
						v.New.Subscribers[newSub] = true
					}
				}

				v.done <- v.New
				close(v.done)
			case ValueStatus:
				val := p.updateInternalStatus(v)
				for sub, _ := range p.Subscribers {
					sub <- val
				}
			}
		case ch := <-p.Unsubscribe:
			delete(p.Subscribers, ch)
		}
	}
}

func (p *Tracker) NewChild() *Tracker {
	if p == nil {
		return nil
	}
	prog := NewTracker(p.Timeout)

	done := make(chan *Tracker)
	// prepare to rcv subs
	defer func() { go prog.Go() }()
	p.Track(ChildStatus{New: prog, done: done})

	return <-done
}

func (t *Tracker) updateInternalStatus(v ValueStatus) (delta ValueStatus) {
	defer func() {
		if delta.Delta != 0 && delta.Delta != 1 && delta.Elapsed == 0 {
			panic("where delta > 0, elapsed time must be specified")
		}
	}()

	delta.Type = v.Type
	delta.Elapsed = time.Since(t.last)
	if v.Elapsed != 0 {
		delta.Elapsed = v.Elapsed
	}

	t.last = time.Now()

	if t.text != v.Text {
		t.text = v.Text
		delta.Text = t.text
	}

	if t.max != v.Max {
		t.max = v.Max
		delta.Max = t.max
	}

	// preference over specified delta
	if v.Value != 0 && t.value != v.Value {
		delta.Delta = v.Value - t.value
		if delta.Delta < 0 {
			panic(fmt.Sprintf("%+v %d %d %s", delta, v.Value, t.value, t.text))
		}
		t.value = v.Value
		delta.Value = t.value
		return
	}

	// specified delta, unspecified new value
	if v.Delta != 0 {
		delta.Delta = v.Delta
		t.value += v.Delta
		delta.Value = t.value
	}

	return

}

func NewTracker(timeout time.Duration) *Tracker {
	return &Tracker{
		ChSubscribe: make(chan chan<- Status),
		Unsubscribe: make(chan chan<- Status),

		Subscribers: make(map[chan<- Status]bool),

		ChTrack: make(chan Status),

		Timeout: timeout,
	}
}
