# IAM Policies for Grid Router

resource "aws_iam_role" "grid_router" {
  name = "grid_router"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF

}

resource "aws_iam_instance_profile" "grid_router" {
  name = "grid_router"
  role = aws_iam_role.grid_router.name
}

# Add a cloudwatch policy for mon-scripts to post to CloudWatch
# https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/mon-scripts.html
resource "aws_iam_policy" "cloudwatch_logs" {
  name        = "grid_router_cloudwatch_logs"
  path        = "/"
  description = "A policy for writing instance data (memory utilization, etc) to CloudWatch"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1535734359418",
      "Action": [
        "cloudwatch:GetMetricStatistics",
        "cloudwatch:ListMetrics",
        "cloudwatch:PutMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "Stmt1535734422775",
      "Action": [
        "ec2:DescribeTags"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF

}

# Add a policy for the instance to be able to send its logs to Cloudwatch Logs
resource "aws_iam_policy" "app_logging" {
  name        = "grid_router_cloudwatch_app_logs"
  path        = "/"
  description = "A policy for writing application logs to Cloudwatch Logs"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1555371707358",
      "Action": [
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF

}

# A policy for Cloudwatch Agent which reports logs to Cloudwatch Logs on the instance
data "aws_iam_policy" "CloudWatchAgentServerPolicy" {
  arn = "arn:aws:iam::aws:policy/CloudWatchAgentServerPolicy"
}

# Attach the cloudwatch policy to the role
resource "aws_iam_role_policy_attachment" "cloudwatch_attach" {
  role       = aws_iam_role.grid_router.name
  policy_arn = aws_iam_policy.cloudwatch_logs.arn
}

# Attach the app_logging policy to the role
resource "aws_iam_role_policy_attachment" "app_logging_attach" {
  role       = aws_iam_role.grid_router.name
  policy_arn = aws_iam_policy.app_logging.arn
}

# Attach the cloudwatch agent logging policy to the role
resource "aws_iam_role_policy_attachment" "cloudwatch_agent_attach" {
  role       = aws_iam_role.grid_router.name
  policy_arn = data.aws_iam_policy.CloudWatchAgentServerPolicy.arn
}

# Attach a ReadOnly ASG policy so that the instances can fetch their ASG's info
resource "aws_iam_role_policy_attachment" "asg_readonly" {
  role       = aws_iam_role.grid_router.name
  policy_arn = "arn:aws:iam::aws:policy/AutoScalingReadOnlyAccess"
}

# Allow AWS SSM Access
resource "aws_iam_role_policy_attachment" "ssm" {
  role       = aws_iam_role.grid_router.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}

# Add a policy for the instance to be able to send its logs to Cloudwatch Logs
resource "aws_iam_policy" "instance_stats" {
  name        = "grid_router_instance_stats"
  path        = "/"
  description = "A policy to allow writing Instance Stats for Beanstalk. https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced.html?icmpid=docs_elasticbeanstalk_console#health-enhanced-authz"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "ElasticBeanstalkHealthAccess",
      "Action": [
        "elasticbeanstalk:PutInstanceStatistics"
      ],
      "Effect": "Allow",
      "Resource": [
        "arn:aws:elasticbeanstalk:*:*:application/*",
        "arn:aws:elasticbeanstalk:*:*:environment/*"
      ]
    }
  ]
}
EOF
}

# Attach the Instance Stat Policy
resource "aws_iam_role_policy_attachment" "instance_stats" {
  role       = aws_iam_role.grid_router.name
  policy_arn = aws_iam_policy.instance_stats.arn
}

# policy that allows your role to assume your sandstorm role
data "aws_iam_policy_document" "sandstorm" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]
    resources = [
      "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/grid_router",
    ]
  }
}

resource "aws_iam_role_policy" "sandstorm_policy" {
  name   = "grid-router-sandstorm-policy"
  role   = aws_iam_role.grid_router.id
  policy = data.aws_iam_policy_document.sandstorm.json
}

// Service Role used for the Beanstalk Environments
resource "aws_iam_role" "grid_router_service_role" {
  name = "grid_router_service_role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "elasticbeanstalk.amazonaws.com"
      },
      "Action": "sts:AssumeRole",
      "Condition": {
        "StringEquals": {
          "sts:ExternalId": "elasticbeanstalk"
        }
      }
    }
  ]
}
EOF
}

// Attach the AWSElasticBeanstalkService policy to the service role
resource "aws_iam_role_policy_attachment" "eb_service_role_attachment" {
  role       = aws_iam_role.grid_router_service_role.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSElasticBeanstalkService"
}
