# Terraform Configuration for Grid Router

# Create a Beanstalk Application
resource "aws_elastic_beanstalk_application" "grid_router" {
  name        = "grid-router"
  description = "A reverse proxy to route requests to Cross Browser Grid"

  appversion_lifecycle {
    service_role          = data.aws_iam_role.beanstalk_service_role.arn
    max_count             = 100
    delete_source_from_s3 = true
  }
}

# Create Beanstalk Environments
module "dev" {
  source                             = "./modules/eb"
  application_name                   = aws_elastic_beanstalk_application.grid_router.name
  instance_profile_arn               = aws_iam_instance_profile.grid_router.arn
  environment                        = "dev"
  sns_arn                            = aws_sns_topic.grid_router.arn
  env_health_threshold               = "30" // Don't alarm, it's staging...
  instance_type                      = "t3.micro"
  asg_min_size                       = "1"
  asg_max_size                       = "1"
  rolling_update_enabled             = "false"     # it's staging - no need
  rolling_update_type                = ""          # it's staging - no need
  deployment_policy                  = "AllAtOnce" # it's staging - no need
  redis_subnet_group                 = aws_elasticache_subnet_group.redis.name
  redis_password                     = var.redis_password
  instance_security_groups           = [data.aws_security_group.artifactory.id, aws_security_group.HubVPCConnections.id]
  subnets                            = [data.aws_subnet.private_a.id]
  service_role                       = aws_iam_role.grid_router_service_role.arn
  duration_to_find_hub_alarm_enabled = false
  nexus_dns_name                     = "nexus-Nexus-1CHA04RG00QFG-873668687.us-west-2.elb.amazonaws.com"
}

output "dev-cname" {
  value = module.dev.cname
}

# Staging - Auto Integration Environments (for PR Testing)
module "dev_ato" {
  source                 = "./modules/eb"
  application_name       = aws_elastic_beanstalk_application.grid_router.name
  instance_profile_arn   = aws_iam_instance_profile.grid_router.arn
  environment            = "dev-ato"
  sns_arn                = aws_sns_topic.grid_router.arn
  env_health_threshold   = "30" // Don't alarm, it's staging...
  instance_type          = "t3.micro"
  asg_min_size           = "1"
  asg_max_size           = "1"
  rolling_update_enabled = "false"     # it's staging - no need
  rolling_update_type    = ""          # it's staging - no need
  deployment_policy      = "AllAtOnce" # it's staging - no need
  redis_subnet_group     = aws_elasticache_subnet_group.redis.name
  redis_password         = var.redis_password
  instance_security_groups = [data.aws_security_group.artifactory.id, aws_security_group.HubVPCConnections.id]
  subnets                  = [data.aws_subnet.private_a.id]
  service_role             = aws_iam_role.grid_router_service_role.arn
  nexus_dns_name           = "nexus-Nexus-1K73POW9851WS-1427429147.us-west-2.elb.amazonaws.com"
  duration_to_find_hub_alarm_enabled = false
}

output "dev-ato_dns" {
  value = module.dev_ato.xarth_dns_name
}

module "prod" {
  source               = "./modules/eb"
  application_name     = aws_elastic_beanstalk_application.grid_router.name
  instance_profile_arn = aws_iam_instance_profile.grid_router.arn
  environment          = "prod"
  acm_override_arn     = aws_acm_certificate.grid_prod_cert.arn
  sns_arn              = aws_sns_topic.grid_router.arn
  env_health_threshold = "20" // Degraded Level
  instance_type        = "t3.large"
  asg_min_size         = "2"
  asg_max_size         = "4"
  deployment_policy    = "Rolling"
  rolling_update_type  = "Health"
  redis_subnet_group   = aws_elasticache_subnet_group.redis.name
  redis_password       = var.redis_password
  instance_security_groups = [data.aws_security_group.artifactory.id, aws_security_group.HubVPCConnections.id]
  subnets                  = [data.aws_subnet.private_a.id, data.aws_subnet.private_b.id, data.aws_subnet.private_c.id]
  service_role             = aws_iam_role.grid_router_service_role.arn
  nexus_dns_name           = "nexus-Nexus-GCXQHJAEWHP5-454884336.us-west-2.elb.amazonaws.com"
}

output "prod-cname" {
  value = module.prod.cname
}

module "prod-2xl" {
  source                   = "./modules/eb"
  application_name         = aws_elastic_beanstalk_application.grid_router.name
  instance_profile_arn     = aws_iam_instance_profile.grid_router.arn
  environment              = "prod-2xl"
  sns_arn                  = aws_sns_topic.grid_router.arn
  env_health_threshold     = "20" // Degraded Level
  instance_type            = "t3.small"
  asg_min_size             = "1"
  asg_max_size             = "1"
  redis_subnet_group       = aws_elasticache_subnet_group.redis.name
  redis_password           = var.redis_password
  instance_security_groups = [data.aws_security_group.artifactory.id, aws_security_group.HubVPCConnections.id]
  subnets                  = [data.aws_subnet.private_a.id, data.aws_subnet.private_b.id]
  service_role             = aws_iam_role.grid_router_service_role.arn
  nexus_dns_name           = "nexus-Nexus-RFP21T8SUDTR-620214493.us-west-2.elb.amazonaws.com"
}

output "prod-2xl-cname" {
  value = module.prod-2xl.cname
}

module "prod-cdp" {
  source                   = "./modules/eb"
  application_name         = aws_elastic_beanstalk_application.grid_router.name
  instance_profile_arn     = aws_iam_instance_profile.grid_router.arn
  environment              = "prod-cdp"
  sns_arn                  = aws_sns_topic.grid_router.arn
  env_health_threshold     = "20"
  instance_type            = "t3.small"
  asg_min_size             = "1"
  asg_max_size             = "1"
  redis_subnet_group       = aws_elasticache_subnet_group.redis.name
  redis_password           = var.redis_password
  instance_security_groups = [data.aws_security_group.artifactory.id, aws_security_group.HubVPCConnections.id]
  subnets                  = [data.aws_subnet.private_a.id, data.aws_subnet.private_b.id, data.aws_subnet.private_c.id]
  service_role             = aws_iam_role.grid_router_service_role.arn
}

output "prod-cdp-cname" {
  value = module.prod-cdp.cname
}
