# Resources required for Jarvis & Access Log Writing

variable "access_log_account_ids" { // https://docs.aws.amazon.com/elasticloadbalancing/latest/classic/enable-access-logs.html#attach-bucket-policy
  type = map(string)
  default = {
    "us-east-1"       = "127311923021"
    "us-east-2"       = "033677994240"
    "us-west-1"       = "027434742980"
    "us-west-2"       = "797873946194"
    "ca-central-1"    = "985666609251"
    "eu-central-1"    = "054676820928"
    "eu-west-1"       = "156460612806"
    "eu-west-2"       = "652711504416"
    "eu-west-3"       = "009996457667"
    "eu-north-1"      = "897822967062"
    "ap-east-1"       = "754344448648"
    "ap-northeast-1"  = "582318560864"
    "ap-northeast-2"  = "600734575887"
    "ap-northeast-3"  = "383597477331"
    "ap-southeast-1"  = "114774131450"
    "ap-southeast-2"  = "783225319266"
    "ap-south-1"      = "718504428378"
    "me-south-1"      = "076674570225"
    "sa-east-1"       = "507241528517"
    "us-gov-west-1"   = "048591011584"
    "us-gov-east-1"   = "190560391635"
    "cn-north-1"      = "638102146993"
    "cn-northwest-1"  = "037604701340"

  }
}

data "aws_region" "current" {} // for fetching the current region

// Bucket to store access logs per environment
resource "aws_s3_bucket" "access_logs" {
  bucket = "${var.application_name}-${var.environment}-access-logs"
  acl    = "private"

  lifecycle_rule {
    id      = "logs"
    enabled = true

    expiration { // jarvis looks up to 14 days back max. Cloudwatch Minimum is 30
      days = 14
    }
  }
}

resource "aws_s3_bucket_public_access_block" "access_logs" { // block all public access
  bucket = aws_s3_bucket.access_logs.id

  block_public_acls       = true
  block_public_policy     = true
  ignore_public_acls      = true
  restrict_public_buckets = true

  depends_on = [ // prevent "A conflicting conditional operation" https://stackoverflow.com/a/60573694/2880393
    aws_s3_bucket_policy.access_logs
  ]
}

// Policy to allow AWS to place logs into the bucket, based on region
resource "aws_s3_bucket_policy" "access_logs" {
  bucket = aws_s3_bucket.access_logs.id

  policy = <<-POLICY
  {
    "Version": "2012-10-17",
    "Statement": [
      {
        "Effect": "Allow",
        "Principal": {
          "AWS": "arn:aws:iam::${var.access_log_account_ids[data.aws_region.current.name]}:root"
        },
        "Action": "s3:PutObject",
        "Resource": "arn:aws:s3:::${aws_s3_bucket.access_logs.id}/*"
      }
    ]
  }
  POLICY
}
