variable "application_name" {}
variable "redis_subnet_group" {}

# https://docs.aws.amazon.com/elasticbeanstalk/latest/platforms/platforms-supported.html#platforms-supported.docker
variable "solution_stack_name" {
  default = "64bit Amazon Linux 2018.03 v2.16.5 running Docker 19.03.13-ce"
}
variable "instance_profile_arn" {}
variable "environment" {}
variable "sns_arn" {}
variable "instance_type" {
  default = "t3.medium"
}

variable "asg_min_size" {
  default = "1"
}

variable "asg_max_size" {
  default = "4"
}

variable "rolling_update_enabled" { # true, false
  default = "true"
}

variable "rolling_update_type" { # Time, Health, Immutable
  default = "Immutable"
}

variable "deployment_policy" { # AllAtOnce, Rolling, RollingWithAdditionalBatch, Immutable
  default = "Immutable"
}

/*
The threshold of the Beanstalk Environment at which it alarms
0 (Ok), 1 (Info), 5 (Unknown), 10 (No data), 15 (Warning), 20 (Degraded), 25 (Severe)
*/
variable "env_health_threshold" { // The threshold of the Beanstalk Environment at which it alarms
  default = "20" // 0 (Ok), 1 (Info), 5 (Unknown), 10 (No data), 15 (Warning), 20 (Degraded), 25 (Severe)
}

# Gather security groups / subnets
data "terraform_remote_state" "vpc" {
  backend = "s3"
  config = {
    profile = "twitch-cape-qe-aws"
    bucket  = "tf-state-qe"
    key     = "tfstate/qe/terraform/vpc"
    region  = "us-west-2"
  }
}

# Create a config template
resource "aws_elastic_beanstalk_configuration_template" "grid_router" {
  name                = "grid-router-${var.environment}-config"
  application         = var.application_name
  solution_stack_name = var.solution_stack_name

  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = data.terraform_remote_state.vpc.outputs.vpc_id
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = join(",", var.subnets)
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "ELBSubnets"
    value     = join(",", var.subnets)
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "AssociatePublicIpAddress"
    value = "false"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "ELBScheme"
    value = "internal"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "InstanceType"
    value = var.instance_type
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MinSize"
    value = var.asg_min_size
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MaxSize"
    value = var.asg_max_size
  }

  # Allow anyone on Twitch CIDRs to ssh into machine
  # SG: allow_ssh_from_twitch_subnets
  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "SecurityGroups"
    value     = join(",", var.instance_security_groups)
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "IamInstanceProfile"
    value = var.instance_profile_arn
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "EC2KeyName"
    value = "grid"
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name = "ServiceRole"
    value = var.service_role
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name = "SystemType"
    value = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application"
    name = "Application Healthcheck URL"
    value = "HTTPS:443/healthcheck"
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment:process:default"
    name = "HealthCheckPath"
    value = "/healthcheck"
  }

  setting {
    namespace = "aws:elb:listener"
    name = "ListenerEnabled"
    value = "true"
  }

  // Port 443 Options
  setting {
    namespace = "aws:elb:listener:443"
    name = "ListenerProtocol"
    value = "HTTPS"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstancePort"
    value = "443"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstanceProtocol"
    value = "HTTPS"
  }

  # If user did not override the acm arn, use the one created in this module
  setting {
    namespace = "aws:elb:listener:443"
    name = "SSLCertificateId"
    value = var.acm_override_arn == "" ? aws_acm_certificate.xarth_cert_nexus[0].arn : var.acm_override_arn
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name = "SecurityGroups"
    value = data.terraform_remote_state.vpc.outputs.twitch_subnets_sg
  }

  setting {
    namespace = "aws:elb:policies"
    name = "ConnectionDrainingEnabled"
    value = "true"
  }

  # Requests can live for up to Grid's BROWSER_TIMEOUT (300s)
  setting {
    namespace = "aws:elb:policies"
    name = "ConnectionDrainingTimeout"
    value = "310"
  }

  # Requests can live for up to Grid's BROWSER_TIMEOUT (300s)
  setting {
    namespace = "aws:elb:policies"
    name = "ConnectionSettingIdleTimeout"
    value = "310"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "RollingUpdateEnabled"
    value = var.rolling_update_enabled
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "RollingUpdateType"
    value = var.rolling_update_type
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "DeploymentPolicy"
    value = var.deployment_policy
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSizeType"
    value = "Percentage"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSize"
    value = "50"
  }

  ############################
  # Auto Scaling Configuration
  ############################
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "MeasureName"
    value = "CPUUtilization"
  }
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "Unit"
    value = "Percent"
  }
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "UpperThreshold"
    value = "70"
  }
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "LowerThreshold"
    value = "35"
  }

  #######################
  # Environment Variables
  #######################
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "REDIS_HOST"
    value = aws_elasticache_replication_group.redis.primary_endpoint_address
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "REDIS_PORT"
    value = aws_elasticache_replication_group.redis.port
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "GR_LOCAL"
    value = "false"
  }
}

# Create the environment
resource "aws_elastic_beanstalk_environment" "grid_router" {
  name                = "grid-router-${var.environment}"
  application         = var.application_name
  template_name       = aws_elastic_beanstalk_configuration_template.grid_router.name
}

# Create a VPC Endpoint to that environment
module "vpc_endpoint" {
  source                = "../vpc_endpoint"
  environment_name      = "grid-router-${var.environment}"
  loadbalancer_dns_name = aws_elastic_beanstalk_environment.grid_router.endpoint_url
  vpc_id                = data.terraform_remote_state.vpc.outputs.vpc_id
  subnets = [
    data.terraform_remote_state.vpc.outputs.private_a,
    data.terraform_remote_state.vpc.outputs.private_b,
    data.terraform_remote_state.vpc.outputs.private_c
  ]
}

# Expose the CNAME
output "cname" {
  value = aws_elastic_beanstalk_environment.grid_router.cname
}
