# Creates a VPC Endpoint connected to a Load Balancer

variable "environment_name" {} # The name of the environment
variable "loadbalancer_dns_name" {} # The DNS Name of the Load Balancer
variable "vpc_id" { # The VPC ID to launch resources into. This should be the same as the Load Balancer ELB
  type = string
}
variable "subnets" { # The Subnets to make the NLB accessible on
  type = list(string)
}

data "aws_region" "current" {} // for fetching the current region

# Create a Lambda Function that will register the ALB IPs to the NLB
resource "aws_lambda_function" "PrivateLinkLambda" {
  function_name = "${var.environment_name}-PrivateLinkLambda"
  s3_bucket     = var.LambdaCodeLocation[data.aws_region.current.name].bucket
  s3_key        = var.LambdaCodeLocation[data.aws_region.current.name].key
  description   = "Register Application Load Balancer to Network Load Balancer"
  handler       = "populate_NLB_TG_with_ALB.lambda_handler"
  role          = aws_iam_role.PrivateLinkLambda.arn
  runtime       = "python2.7"
  timeout       = 300

  environment {
    variables = {
      ALB_DNS_NAME              = var.loadbalancer_dns_name
      ALB_LISTENER              = 443
      CW_METRIC_FLAG_IP_COUNT   = false
      MAX_LOOKUP_PER_INVOCATION = 50
      NLB_TG_ARN                = aws_lb_target_group.PrivateLink.arn
      S3_BUCKET                 = aws_s3_bucket.ELBIPsBucket.id
      INVOCATIONS_BEFORE_DEREGISTRATION = 3
    }
  }

  tags = {
    "elasticbeanstalk:environment-name" = var.environment_name
  }
}

# Run the above Lambda every minute
resource "aws_cloudwatch_event_rule" "PrivateLinkLambda" {
  name        = "${var.environment_name}-PrivateLink"
  description = "Trigger Lambda to register NLB Targets"
  schedule_expression = "rate(1 minute)"
  is_enabled = true

  tags = {
    "elasticbeanstalk:environment-name" = var.environment_name
  }
}

resource "aws_cloudwatch_event_target" "PrivateLinkLambda" {
  rule      = aws_cloudwatch_event_rule.PrivateLinkLambda.name
  target_id = "TargetFunctionV1"
  arn       = aws_lambda_function.PrivateLinkLambda.arn
}

resource "aws_lambda_permission" "PrivateLinkLambda" {
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.PrivateLinkLambda.function_name
  principal     = "events.amazonaws.com"
  source_arn    = aws_cloudwatch_event_rule.PrivateLinkLambda.arn
}

# Store Logs
resource "aws_cloudwatch_log_group" "launcher" {
  name = "/aws/lambda/${aws_lambda_function.PrivateLinkLambda.function_name}"
  retention_in_days = 30

  tags = {
    "elasticbeanstalk:environment-name" = var.environment_name
  }
}

# IAM Permissions for the Lambda
resource "aws_iam_role" "PrivateLinkLambda" {
  description = "Used for Lambda Private Link Registration"

  tags = {
    "elasticbeanstalk:environment-name" = var.environment_name
  }

  assume_role_policy = <<-EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "PrivateLinkLambda" {
  name = "Lambda-ALBasTarget"
  role = aws_iam_role.PrivateLinkLambda.name
  policy = <<-EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": [
                "logs:CreateLogStream",
                "logs:PutLogEvents"
            ],
            "Resource": [
                "${aws_cloudwatch_log_group.launcher.arn}"
            ],
            "Effect": "Allow",
            "Sid": "LambdaLogging"
        },
        {
            "Action": [
                "s3:Get*",
                "s3:PutObject",
                "s3:CreateBucket",
                "s3:ListBucket",
                "s3:ListAllMyBuckets"
            ],
            "Resource": "*",
            "Effect": "Allow",
            "Sid": "S3"
        },
        {
            "Action": [
                "elasticloadbalancing:Describe*",
                "elasticloadbalancing:RegisterTargets",
                "elasticloadbalancing:DeregisterTargets"
            ],
            "Resource": "*",
            "Effect": "Allow",
            "Sid": "ELB"
        },
        {
            "Action": [
                "cloudwatch:putMetricData"
            ],
            "Resource": "*",
            "Effect": "Allow",
            "Sid": "CW"
        }
    ]
}
EOF
}

# S3 Bucket for storing the ELB IPs
resource "aws_s3_bucket" "ELBIPsBucket" {
  bucket = "${var.environment_name}-elb-ips-bucket"
  force_destroy = true

  tags = {
    "elasticbeanstalk:environment-name" = var.environment_name
  }
}

# NLB that is used to forward the requests from the VPC Endpoint Private Link to the Application LB
resource "aws_lb" "NLB" {
  name = "${var.environment_name}-PLELB"
  internal = true
  load_balancer_type = "network"
  subnets = var.subnets
  enable_cross_zone_load_balancing = true
  enable_deletion_protection       = true

  tags = {
    Name = "${var.environment_name}-PLELB"
    "elasticbeanstalk:environment-name" = var.environment_name
  }
}

# Target to the Application LB, which will be updated by the PrivateLinkLambda Function
resource "aws_lb_target_group" "PrivateLink" {
  name        = "${var.environment_name}-PLELB-https"
  port        = 443
  protocol    = "TCP"
  target_type = "ip"
  vpc_id      = var.vpc_id
  deregistration_delay = 300

  health_check {
    protocol            = "TCP"
    interval            = 30
    healthy_threshold   = 3
    unhealthy_threshold = 3
  }

  tags = {
    "elasticbeanstalk:environment-name" = var.environment_name
  }
}

resource "aws_lb_listener" "PrivateLink" {
  load_balancer_arn = aws_lb.NLB.arn
  port              = 443
  protocol          = "TCP"

  default_action {
    type = "forward"
    target_group_arn = aws_lb_target_group.PrivateLink.arn
  }
}

# VPC Endpoint for consumers
resource "aws_vpc_endpoint_service" "PrivateLink" {
  network_load_balancer_arns = [aws_lb.NLB.arn]
  acceptance_required        = true
  allowed_principals         = ["*"]

  lifecycle {
    prevent_destroy = true
  }

  tags = {
    Name                                = var.environment_name
    "elasticbeanstalk:environment-name" = var.environment_name
  }
}

output "vpc_endpoint" {
  value = aws_vpc_endpoint_service.PrivateLink.service_name
}
