#!/bin/bash

# List of Account IDs to publish artifacts to
accountsIDsToPublishArtifact=(
  "280017927569" # twitch-browsergrid-ui-dev
  "276846748787" # twitch-browsergrid-ui-prod
)

# Variables used through the script
applicationName="grid-ui"
version="$GIT_COMMIT"
s3Path="apps/$applicationName/$version.zip"
description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
region="us-west-2"

main() {
  manta -v
  cp -R .ebextensions beanstalk/

  export AWS_STS_REGIONAL_ENDPOINTS=regional

  # Zip Contents
  cd beanstalk
  zip -r ./${version}.zip ./

  # Create artifacts for each account
  for accountID in ${accountsIDsToPublishArtifact[@]}; do
    createArtifactInAccount "$accountID"
  done
}

# Assumes an AWS Role
# @param Account ID to assume
assumeRole() {
  # Unset to prevent old credentials being used
  unset AWS_SECRET_ACCESS_KEY
  unset AWS_SECRET_KEY
  unset AWS_SESSION_TOKEN

  local assumeRoleARN="$1"
  echo "Assuming Role: $assumeRoleARN"
  output=$(aws sts assume-role --role-arn "$assumeRoleARN" --role-session-name "Jenkins-Session")

  if [ $? -ne 0 ]; then
    echo "Error assuming role: $assumeRoleARN"
    exit 1
  fi

  # Set Credentials
  export AWS_ACCESS_KEY_ID=$(echo $output | jq -r '.Credentials.AccessKeyId')
  export AWS_SECRET_ACCESS_KEY=$(echo $output | jq -r '.Credentials.SecretAccessKey')
  export AWS_SESSION_TOKEN=$(echo $output | jq -r '.Credentials.SessionToken')

  aws sts get-caller-identity
}

# Resets AWS Credentials, which helps between assuming roles to ensure your assumed-role isn't assuming another role
resetAWSCredentials() {
  unset AWS_ACCESS_KEY_ID
  unset AWS_SECRET_ACCESS_KEY
  unset AWS_SESSION_TOKEN
}

# Creates a Beanstalk Artifact in an account
# @param Account ID to upload artifact into
createArtifactInAccount() {
  local accountID="$1"
  local assumeRoleARN="arn:aws:iam::$accountID:role/jenkins"
  echo "Creating Beanstalk Application Version in Account ID $accountID"

  assumeRole "$assumeRoleARN"

  # Upload the artifacts into S3
  local s3Bucket="elasticbeanstalk-$region-$accountID"
  aws s3 cp --quiet "$version.zip" "s3://$s3Bucket/$s3Path"

  # Create an application version
  aws elasticbeanstalk create-application-version --no-auto-create-application \
   --application-name "$applicationName" \
   --version-label "$version" \
   --description "$description" \
   --source-bundle S3Bucket="$s3Bucket",S3Key="$s3Path"

   if [ $? -ne 0 ]; then # fail build on error
    exit 1
   fi

   resetAWSCredentials
}

main
