import * as React from "react"
import { connect } from "react-redux";
import { UncontrolledDropdown, DropdownToggle, DropdownMenu, DropdownItem } from "reactstrap"
import { bindActionCreators, Dispatch } from "redux";
import updateEnvironment from "../store/environment/actions"
import { AppState } from "../store/store";
import { EnvironmentState } from "../store/environment/types";

// Stores all of the Grid environments and their URL
export const Environments: {[index: string]:string} = {
  "Production":  "https://api-prod-nexus.browsergrid.xarth.tv",
  "Performance": "https://api-prod-2xl-nexus.browsergrid.xarth.tv",
  "Staging":     "https://api-dev-nexus.browsergrid.xarth.tv",
  "Integration": "https://api-dev-ato-nexus.browsergrid.xarth.tv"
};

// Props specific for the EnvironmentDropdown
interface EnvironmentDropdownProps { }

// State specific for the EnvironmentDropDown
interface EnvironmentDropdownState { }

// Redux props
interface LinkStateProps {
  environment: EnvironmentState
}

// Redux Dispatch
interface LinkDispatchProps {
  updateEnvironment: (newEnvironment: string, newUrl: string) => void;
}

// Combined Props
type Props = EnvironmentDropdownProps & LinkDispatchProps & LinkStateProps;

export class EnvironmentDropdown extends React.Component<Props, EnvironmentDropdownState> {
  constructor(props: Props) {
    super(props);

    // This binding is necessary to make `this` work in the callback
    this.selectDropdown = this.selectDropdown.bind(this);
  }

  // Function that runs when an environment dropdown item is selected
  // event: The mouse event passed from react
  selectDropdown(event: React.MouseEvent) {
    let textContent = String(event.currentTarget.textContent);
    this.props.updateEnvironment(textContent, this.getAPIURL(textContent));
  }

  // Returns the Grid API URL from the environments
  // env: The environment name
  getAPIURL(env: string): string {
    return Environments[env]
  }

  render() {
    let envDropdownItems = Object.keys(Environments).map((envName: string) => {
      return (
        <DropdownItem key={envName} onClick={this.selectDropdown}>{envName}</DropdownItem>
      )
    });

    return (
        <UncontrolledDropdown nav inNavbar>
          <DropdownToggle nav caret>{this.props.environment.name}</DropdownToggle>
          <DropdownMenu right>
            <DropdownItem header>Environment</DropdownItem>
            {envDropdownItems}
          </DropdownMenu>
        </UncontrolledDropdown>
    )
  }
}

function mapStateToProps(state: AppState): LinkStateProps { // Redux
  return {
    environment: state.environment
  }
}

function mapDispatchToProps(dispatch: Dispatch): LinkDispatchProps { // Redux
  return {
    updateEnvironment: bindActionCreators(updateEnvironment, dispatch),
  }
}

// Connect to Redux
export default connect(mapStateToProps, mapDispatchToProps)(EnvironmentDropdown)
