import * as React from "react"
import axios from "axios";
import { Button, Progress } from "reactstrap";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faPause, faPlay } from "@fortawesome/free-solid-svg-icons";
import {PauseHub, UnpauseHub} from "../api/hubs";

interface Props {
  hub: Hub, // the Hub item to display
  environment: {
    url: string, // the grid environment url to make requests to
  },
  displayActions: boolean, // whether actions should be displayed, which is an admin only function
  updateHubHandler: Function, // trigger to update hubs
  nexusToken: string,
}

interface State { }

// Hub object returned from Grid Router
export interface Hub {
  id: string,
  Hostname: string,
  SlotCounts: {
    Free: number,
    Total: number,
  },
  paused: boolean,
  healthy: boolean,
}

export class HubItem extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);
  }

  // Pauses a hub in the hub registry
  pause(hub: Hub) {
    PauseHub(this.props.environment.url, this.props.nexusToken, hub).then(() => {
      this.props.updateHubHandler() // update the items to get the paused state
    })
  }

  // Unpause a hub in the hub registry
  unpause(hub: Hub) {
    UnpauseHub(this.props.environment.url, this.props.nexusToken, hub).then(() => {
      this.props.updateHubHandler() // update the items to get the unpaused state
    })
  }

  // Renders a progress bar that shows hub capacity
  renderProgressBar(hub: Hub) {
    let slotsFree = hub.SlotCounts.Free;
    let slotsTotal = hub.SlotCounts.Total;
    let inUse = slotsTotal - slotsFree;
    let utilPercent = (inUse / slotsTotal) * 100;
    let color = "success";

    // Determine the color of the percent bar
    if (utilPercent >= 90) {
      color = "danger"
    } else if (utilPercent >= 60) {
      color = "warning"
    } else {
      color = "success"
    }

    return (
      <Progress color={color} value={inUse} max={slotsTotal}>
        {(slotsTotal - slotsFree) + "/" + slotsTotal}
      </Progress>
    )
  }

  // Renders the pause button within the actions column
  renderPauseButton(hub: Hub) {
    if (!this.props.displayActions) {
      return
    }

    if (hub.paused) {
      return (
        <td>
          <Button size="sm" outline color="info" onClick={this.unpause.bind(this, hub)}>
            <FontAwesomeIcon icon={faPlay}/>
          </Button>
        </td>
      )
    } else {
      return (
        <td>
          <Button color="warning" outline size="sm" onClick={this.pause.bind(this, hub)}>
            <FontAwesomeIcon icon={faPause}/>
          </Button>
        </td>
      )
    }
  }

  render() {
    let rowColor = '';

    if (!this.props.hub.healthy) { // if not healthy
      rowColor = 'table-danger'
    } else if (this.props.hub.paused) { // if paused
      rowColor = 'table-warning'
    }

    return (
      <tr className={rowColor}>
        <td>{this.props.hub.id}</td>
        <td>{this.props.hub.Hostname}</td>
        <td id="utilizationCol">{this.renderProgressBar(this.props.hub)}</td>
        {this.renderPauseButton(this.props.hub)}
      </tr>
    );
  }
}
