import * as React from "react"
import "./hubs.scss"
import { Table, Progress, Container, Row, Col, Button } from "reactstrap";
import axios from "axios";
import { connect } from "react-redux";
import { AppState } from "../store/store";
import { EnvironmentState } from "../store/environment/types";
import queryString from 'query-string'
import { HubItem, Hub } from "./hubItem";
import { GetHubs } from "../api/hubs";
import { GetNexusToken } from "../api/nexus";

// Props specific for the Hub Class
type HubProps = {
  location: {
    pathname: string,
    search: string,
  }
}

// State specific for the Hub class
type HubState = {
  hubs: Hub[],
  nexusToken: string
}

// Props from Redux
interface LinkStateProps {
  environment: EnvironmentState
}

// Props combined with Redux props
type Props = HubProps & LinkStateProps;

export class Hubs extends React.Component<Props, HubState> {
  constructor(props: Props) {
    super(props);

    this.state = {
      hubs: [],
      nexusToken: ""
    };

    this.getHubs = this.getHubs.bind(this)
  }

  // Initialize a variable for the interval to poll
  private interval: number = 0;

  async componentWillMount() {
    await this.updateToken()
    this.getHubs(); // initial fetch of hubs on load

    // Refresh results every 5 seconds
    this.interval = window.setInterval(() => this.getHubs(), 5000);
  }

  componentWillUnmount() {
    window.clearInterval(this.interval); // clear out the interval when changing pages
  }

  componentDidUpdate(prevProps: Props){
    // if the user switches the environment url, immediately fetch new hubs
    if (this.props.environment.url !== prevProps.environment.url) {
      this.getHubs()
    }
  }

  // Determine if the request has an admin query parameter
  isAdmin(): boolean {
    const values = queryString.parse(this.props.location.search);
    return (values["admin"] === "true")
  }

  // Updates the application state with the current Nexus Token
  async updateToken() {
    await GetNexusToken().then(token => {
      this.setState({
        nexusToken: token,
      })
    })
  }

  // Gets all hubs registered with the registry
  async getHubs() {
    GetHubs(this.props.environment.url, this.state.nexusToken).then(resp => {
      if (resp.status === 200) {
        this.setState({
          hubs: resp.data
        })
      } else {
        throw new Error(`Unexpected response status. Response: ${resp}`);
      }
    }).catch(e => {
      if (e.response) { // if the request was made with a server response (non-2xx)
        if (e.response.status === 401) { // Nexus tokens are only valid for 15 minutes. If it's coming back unauthorized, reauth
          console.log("Received a 401. Refetching token")
          this.updateToken()
        }
      } else {
        // Something happened in setting up the request that triggered an Error
        throw(e)
      }
    })
  }

  render() {
    // Create a table row for each hub
    let hubs = this.state.hubs.map((hub: Hub) => {
      return (
        <HubItem key={hub.id} hub={hub} environment={this.props.environment} displayActions={this.isAdmin()}
                 updateHubHandler={this.getHubs} nexusToken={this.state.nexusToken} />
      );
    });

    let actionColumn = () => {
      if (this.isAdmin()) {
        return (
          <th>Actions</th>
        )
      }
    };

    return (
      <Container>
        <Row>
          <Col><h1 className="pageHeader">Hub Registry</h1></Col>
        </Row>
        <Row>
          <Col>
            <Table>
              <thead>
                <tr>
                  <th>ID</th>
                  <th>Hostname</th>
                  <th>Utilization</th>
                  {actionColumn()}
                </tr>
              </thead>
              <tbody>
                {hubs}
              </tbody>
            </Table>
          </Col>
        </Row>
      </Container>
    )
  }
}

function mapStateToProps(state: AppState): LinkStateProps { // Redux
  return {
    environment: state.environment
  }
}

// Connect to Redux
export default connect(mapStateToProps, {})(Hubs)
