variable "application_name" {}
variable "instance_profile_arn" {}

# https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/concepts.platforms.html#concepts.platforms.docker
# https://aws.amazon.com/releasenotes/?tag=releasenotes%23keywords%23aws-elastic-beanstalk
variable "solution_stack_name" {
  default = "64bit Amazon Linux 2018.03 v2.16.8 running Docker 19.03.13-ce"
}
variable "environment" {}
variable "instance_type" {
  default = "t3.micro"
}

variable "asg_min_size" {
  default = "1"
}

variable "asg_max_size" {
  default = "1"
}

variable "rolling_update_enabled" { # true, false
  default = "false"
}

variable "rolling_update_type" { # Time, Health, Immutable
  default = "Immutable"
}

variable "deployment_policy" { # AllAtOnce, Rolling, RollingWithAdditionalBatch, Immutable
  default = "Rolling"
}

# Create a config template
resource "aws_elastic_beanstalk_configuration_template" "grid_ui" {
  name                = "${var.application_name}-${var.environment}-config"
  application         = var.application_name
  solution_stack_name = var.solution_stack_name

  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = var.vpc_id
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = join(",", var.subnets)
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "AssociatePublicIpAddress"
    value = "false"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "ELBScheme"
    value = "internal"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "InstanceType"
    value = var.instance_type
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "IamInstanceProfile"
    value = var.instance_profile_arn
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MinSize"
    value = var.asg_min_size
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MaxSize"
    value = var.asg_max_size
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name = "SystemType"
    value = "enhanced"
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name = "SecurityGroups"
    value = join(",", var.elb_security_groups)
  }

  ############
  # Port 443 #
  ############
  setting {
    namespace = "aws:elb:listener:443"
    name = "ListenerProtocol"
    value = "HTTPS"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstancePort"
    value = "443"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstanceProtocol"
    value = "HTTPS"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "SSLCertificateId"
    value = aws_acm_certificate.xarth_cert.arn
  }

  ##########
  # Others #
  ##########
  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "RollingUpdateEnabled"
    value = var.rolling_update_enabled
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "RollingUpdateType"
    value = var.rolling_update_type
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "DeploymentPolicy"
    value = var.deployment_policy
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSizeType"
    value = "Percentage"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSize"
    value = "50"
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name = "ServiceRole"
    value = var.service_role_arn
  }

  ############################
  # Auto Scaling Configuration
  ############################
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "MeasureName"
    value = "CPUUtilization"
  }
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "Unit"
    value = "Percent"
  }
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "UpperThreshold"
    value = "70"
  }
  setting {
    namespace = "aws:autoscaling:trigger"
    name = "LowerThreshold"
    value = "35"
  }

  #############################
  # Environment Configuration #
  #############################
  setting {
    namespace = "aws:elasticbeanstalk:environment:proxy"
    name = "ProxyServer"
    value = "none"
  }
}

resource "aws_elastic_beanstalk_environment" "grid_ui" {
  name          = "${var.application_name}-${var.environment}"
  application   = var.application_name
  template_name = aws_elastic_beanstalk_configuration_template.grid_ui.name
}

output "cname" {
  value = aws_elastic_beanstalk_environment.grid_ui.cname
}
