package heimdall

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/common/golibs/log"
	"code.justin.tv/foundation/twitchclient"
)

const (
	pollSleep  = 10 * time.Second
	maxRetries = 15
)

// User pool types
const (
	PaymentsAutomation                        string = "PAYMENTS_AUTOMATION"
	PaymentsAutomationNoCOR                   string = "PAYMENTS_AUTOMATION_NO_COR"
	PaymentsAutomationRecurlyAmazonPay        string = "PAYMENTS_AUTOMATION_RECURLY_AMAZON_PAY"
	PaymentsAutomationRecurlyAmazonPayCA      string = "PAYMENTS_AUTOMATION_RECURLY_AMAZON_PAY_CA"
	PaymentsAutomationRecurlyAmazonPayInvalid string = "PAYMENTS_AUTOMATION_RECURLY_AMAZON_PAY_INVALID"
	PaymentsAutomationRecurlyPaypal           string = "PAYMENTS_AUTOMATION_RECURLY_PAYPAL"
	PaymentsAutomationRecurlySPM              string = "PAYMENTS_AUTOMATION_RECURLY_SPM"
	PaymentsAutomationZuoraPaypal             string = "PAYMENTS_AUTOMATION_ZUORA_PAYPAL"
	PaymentsAutomationPartner                 string = "PAYMENTS_AUTOMATION_PARTNER"
	PaymentsAutomationUS                      string = "PAYMENTS_AUTOMATION_US"
	PaymentsAutomationEU                      string = "PAYMENTS_AUTOMATION_EU"

	DeveloperOrganizationOwner  string = "PAYMENTS_DEV_ORG_OWNER"
	DeveloperOrganizationMember string = "PAYMENTS_DEV_ORG_MEMBER"
)

type Heimdall interface {
	Get(ctx context.Context, userType string, environment string) (*http.Response, *GetAvailableUserResponse, error)
	GetAvailableUser(ctx context.Context, userType string, env string) (User, error)
	UnlockUser(ctx context.Context, userType string, username string, env string) (*http.Response, *UnlockUserResponse, error)
	DeferUnlock(ctx context.Context, userType string, user User, env string) error
}

type Client struct {
	heimdall twitchclient.JSONClient
}

func NewHeimdallClient(host string) Heimdall {
	heimdall := newClientService(host)
	return &Client{
		heimdall: heimdall,
	}
}

func newClientService(hostName string) twitchclient.JSONClient {
	conf := twitchclient.ClientConf{}

	twitchClient := twitchclient.NewHTTPClient(conf)
	twitchJSONClient := twitchclient.NewJSONClient(hostName, twitchClient)

	return *twitchJSONClient
}

// GetAvailableUser tries maxRetries times to get an account of userType and locks it for use
func (c *Client) GetAvailableUser(ctx context.Context, userType string, env string) (User, error) {
	user := User{}
	retryCount := 0

	for user.User.Username == "" && retryCount < maxRetries {
		if retryCount > 0 {
			log.Warn("Heimdall attempt #%d of %d", retryCount, maxRetries)
			time.Sleep(pollSleep)
		}

		resp, respBody, err := c.Get(ctx, string(userType), env)
		if err != nil {
			if resp == nil {
				return user, err
			}
			if resp.StatusCode != 422 {
				return user, fmt.Errorf("heimdall.GetAvailableUser(%s) error: %s", userType, err)
			}
		}

		if resp.StatusCode == 200 {
			user.User = respBody.Body
		}
		retryCount++
	}

	if user.User.Username == "" {
		return user, fmt.Errorf("unable to find an available user of type %s", userType)
	}

	err := c.getDetailsFromAuthToken(&user)
	if err != nil {
		return user, fmt.Errorf("getDetailsFromAuthToken(&%v) error: %s", user, err)
	}
	return user, nil
}

// Get retrieves an account of userType and locks it for use
func (c *Client) Get(ctx context.Context, userType string, environment string) (*http.Response, *GetAvailableUserResponse, error) {
	path := fmt.Sprintf("/types/%s/available/%s", userType, environment)

	requestBody := `{}`

	req := twitchclient.JSONRequest{
		Method: "PUT",
		Path:   path,
		Body:   requestBody,
		Header: nil,
	}

	var responseBody *GetAvailableUserResponse
	response, err := c.heimdall.Do(ctx, req, &responseBody)
	return response, responseBody, err
}

// Unlock unlocks a locked user account
func (c *Client) UnlockUser(ctx context.Context, userType string, username string, env string) (*http.Response, *UnlockUserResponse, error) {
	path := fmt.Sprintf("/types/%s/users/%s/%s/unlock", userType, username, env)

	requestBody := `{}`

	req := twitchclient.JSONRequest{
		Method: "PUT",
		Path:   path,
		Body:   requestBody,
		Header: nil,
	}

	var responseBody *UnlockUserResponse
	response, err := c.heimdall.Do(ctx, req, &responseBody)
	return response, responseBody, err
}

// DeferUnlock is a helper to unlock a user after test finishes running
func (c *Client) DeferUnlock(ctx context.Context, userType string, user User, env string) error {
	if user.User.Username != "" {
		_, _, err := c.UnlockUser(ctx, userType, user.User.Username, env)
		if err != nil {
			return fmt.Errorf("error unlocking user %s", user.User.Username)
		}
	}
	return nil
}

// getDetailsFromAuthToken parses the auth token from the Heimdall user response
func (c *Client) getDetailsFromAuthToken(user *User) error {

	s, err := url.QueryUnescape(user.User.AuthenticationToken)
	if err != nil {
		return fmt.Errorf("url.QueryUnescape(%s) error: %s", user.User.AuthenticationToken, err)
	}

	var authorizationToken AuthenticationToken

	if s == "" {
		return fmt.Errorf("user does not have an authentication_token")
	}

	s = strings.Replace(s, `"{`, `{`, 1)
	s = strings.Replace(s, `}"`, `}`, 1)

	err = json.Unmarshal([]byte(s), &authorizationToken)
	if err != nil {
		return fmt.Errorf("json.Unmarshal(%s, &%v) error: %s", []byte(s), authorizationToken, err)
	}

	user.Authorization = fmt.Sprintf("OAuth %s", authorizationToken.AuthToken.AuthToken)
	user.UserID = authorizationToken.TUID
	return nil
}
